// Utility functions

export function sanitizeTitle(title: string): string {
  if (!title) return "";

  let slug = title.toLowerCase();

  const specialChars: { [key: string]: string } = {
    ä: "ae",
    ö: "oe",
    ü: "ue",
    ß: "ss",
    à: "a",
    á: "a",
    â: "a",
    ã: "a",
    å: "a",
    è: "e",
    é: "e",
    ê: "e",
    ë: "e",
    ì: "i",
    í: "i",
    î: "i",
    ï: "i",
    ò: "o",
    ó: "o",
    ô: "o",
    õ: "o",
    ù: "u",
    ú: "u",
    û: "u",
    ñ: "n",
    ç: "c",
    æ: "ae",
    œ: "oe",
  };

  Object.keys(specialChars).forEach((char) => {
    slug = slug.replace(new RegExp(char, "g"), specialChars[char]);
  });

  slug = slug
    .replace(/[^\w\s-]/g, "")
    .replace(/[\s\/\\_]+/g, "-")
    .replace(/-+/g, "-")
    .replace(/^-+|-+$/g, "");

  return slug;
}

export function formatSalary(
  salary: string | null,
  minSalary: number | null,
  maxSalary: number | null,
  currency: string,
  rateSalary: string,
  salryPayBy: "range" | "fixed"
): string {
  if (!salary && !minSalary && !maxSalary) return "";

  const currencySymbols: { [key: string]: string } = {
    USD: "$",
    EUR: "€",
    GBP: "£",
    JPY: "¥",
    CNY: "¥",
    INR: "₹",
    AUD: "A$",
    CAD: "C$",
    CHF: "CHF ",
    SEK: "kr ",
    NOK: "kr ",
    DKK: "kr ",
    PLN: "zł ",
    BRL: "R$",
    MXN: "$",
    ZAR: "R",
    SGD: "S$",
    HKD: "HK$",
    KRW: "₩",
    THB: "฿",
    IDR: "Rp ",
    MYR: "RM ",
    PHP: "₱",
    VND: "₫",
    TRY: "₺",
    AED: "د.إ ",
    SAR: "﷼ ",
    ILS: "₪",
    NZD: "NZ$",
    ARS: "$",
    CLP: "$",
    COP: "$",
    PEN: "S/ ",
    HUF: "Ft ",
    CZK: "Kč ",
    RON: "lei ",
    BGN: "лв ",
    EGP: "£",
    PKR: "₨ ",
    BDT: "৳ ",
    NGN: "₦",
    KES: "KSh ",
    TWD: "NT$",
  };

  const symbol = currencySymbols[currency.toUpperCase()] || `${currency} `;
  const rate = rateSalary || "";

  const formatNumber = (num: number): string => {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
  };

  if (salryPayBy === "range") {
    const min = minSalary ? formatNumber(minSalary) : "";
    const max = maxSalary ? formatNumber(maxSalary) : "";

    if (min && max) {
      return `${symbol}${min} – ${symbol}${max}${rate ? ` / ${rate}` : ""}`;
    } else if (min) {
      return `${symbol}${min}${rate ? ` / ${rate}` : ""}`;
    } else if (max) {
      return `${symbol}${max}${rate ? ` / ${rate}` : ""}`;
    }
  } else {
    const sal = salary ? formatNumber(parseFloat(salary)) : "";
    if (sal) {
      return `${symbol}${sal}${rate ? ` / ${rate}` : ""}`;
    }
  }

  return "";
}

// All supported career domains from CareerDomainEnum
const CAREER_DOMAINS = [
  "apply-link.net",
  "apply-jobs.net",
  "bewerben.link",
  "appliquer.net",
  "postular.link",
  "solliciteren.link",
  "candidarsi.link",
];

export function getSubdomain(): string {
  if (typeof window === "undefined") return "";
  const hostname = window.location.hostname;
  const parts = hostname.split(".");

  // Handle localhost with subdomain (e.g., creativity.localhost:3000)
  if (hostname.includes("localhost")) {
    return parts[0] || "";
  }

  // Handle flowxtra.com domains (e.g., creativity.flowxtra.com)
  if (hostname.endsWith("flowxtra.com")) {
    return parts[0] || "";
  }

  // Handle career domains (e.g., dpro.apply-link.net -> dpro)
  for (const domain of CAREER_DOMAINS) {
    if (hostname.endsWith(domain)) {
      const domainParts = domain.split(".");
      const subdomainParts = parts.slice(0, -domainParts.length);
      if (subdomainParts.length > 0) {
        return subdomainParts[0] || "";
      }
    }
  }

  // Fallback: return first part
  return parts[0] || "";
}

export function getUrlSitemap(subdomain: string): string {
  if (subdomain === "my") {
    return "https://my.flowxtra.com/";
  } else {
    return `https://${subdomain}.flowxtra.com/`;
  }
}

export function getCurrentUrlWithoutProtocol(): string {
  if (typeof window === "undefined") return "";
  // Get full URL and remove http:// or https://
  const fullUrl = window.location.href;
  return fullUrl.replace(/^https?:\/\//, "");
}

export function getCareerDomainFromUrl(): string {
  if (typeof window === "undefined") return "";
  try {
    const hostname = window.location.hostname;
    // Extract career domain from hostname (e.g., creativity.apply-link.net -> apply-link.net)
    const parts = hostname.split(".");
    if (parts.length >= 2) {
      // Take the last two parts (e.g., apply-link.net)
      return parts.slice(-2).join(".");
    }
    return hostname;
  } catch (error) {
    console.error("Error extracting career domain:", error);
    return "";
  }
}
