// Server-side utility functions

export function getOriginalHost(headers: Headers): string {
  const xfHost = headers.get("x-forwarded-host") || "";
  const host = xfHost.split(",")[0].trim() || headers.get("host") || "";
  return host.replace(/:\d+$/, "");
}

export function getProto(headers: Headers): string {
  const xfProto = headers.get("x-forwarded-proto") || "";
  const proto = xfProto.split(",")[0].trim() || "https";
  return proto.replace(/:$/, "");
}

// All supported career domains from CareerDomainEnum
const CAREER_DOMAINS = [
  "apply-link.net",
  "apply-jobs.net",
  "bewerben.link",
  "appliquer.net",
  "postular.link",
  "solliciteren.link",
  "candidarsi.link",
];

export function getSubdomainFromHost(host: string): string | null {
  if (!host) return null;

  const parts = host.split(".");

  // للـ localhost: dpro.localhost:3200 -> dpro
  if (parts.includes("localhost")) {
    if (parts.length >= 2) {
      return parts[0].toLowerCase();
    }
  }

  // للـ flowxtra.com: dpro.flowxtra.com -> dpro
  const root = "flowxtra.com";
  if (host.endsWith(root)) {
    if (parts.length >= 3) {
      return parts[0].toLowerCase();
    }
  }

  // للـ career domains: dpro.apply-link.net -> dpro
  for (const domain of CAREER_DOMAINS) {
    if (host.endsWith(domain)) {
      // Extract subdomain: dpro.apply-link.net -> dpro
      const domainParts = domain.split(".");
      const subdomainParts = parts.slice(0, -domainParts.length);
      if (subdomainParts.length > 0) {
        return subdomainParts[0].toLowerCase();
      }
    }
  }

  return null;
}

export function absolutize(route: string, host: string, proto: string): string {
  try {
    const base = `${proto}://${host}`;
    const u = new URL(route, base);
    u.protocol = `${proto}:`;
    u.host = host;
    u.hostname = host;
    return u.href;
  } catch {
    const clean = String(route || "").startsWith("/")
      ? route
      : `/${route || ""}`;
    return `${proto}://${host}${clean}`;
  }
}

export function isoNoMillis(d: Date): string {
  return d.toISOString().replace(/\.\d{3}Z$/, "Z");
}

export function sha1(s: string): string {
  // Use Node.js crypto module for proper SHA1 hashing
  const crypto = require("crypto");
  return crypto.createHash("sha1").update(s).digest("hex");
}
