// Brand Kit utility functions for Next.js

import type {
  TypographySettings,
  LogoSettings,
  FooterSettings,
  SocialLink,
  SocialStyling,
  Company,
} from "./types";

const SYSTEM_FONTS = [
  "Arial",
  "Times New Roman",
  "Georgia",
  "Helvetica",
  "Verdana",
  "Courier New",
  "Tahoma",
  "Trebuchet MS",
  "Impact",
  "Lucida Console",
  "Lucida Grande",
  "Calibri",
  "Arial Black",
  "Arial Narrow",
  "Century Gothic",
  "Book Antiqua",
  "Cambria",
  "Candara",
  "Corbel",
  "Franklin Gothic Medium",
  "Segoe UI",
];

/**
 * Load Google Fonts dynamically
 */
export function loadTypographyFonts(
  typographySettings: TypographySettings | null
): void {
  if (!typographySettings || typeof window === "undefined") return;

  const headingsFamily =
    typographySettings.headings?.family || typographySettings.headings_family;
  const bodyFamily =
    typographySettings.body?.family || typographySettings.body_family;

  [headingsFamily, bodyFamily].forEach((fontName) => {
    if (fontName && !SYSTEM_FONTS.includes(fontName)) {
      injectGoogleFont(fontName);
    }
  });
}

/**
 * Inject Google Font dynamically
 */
function injectGoogleFont(fontName: string): void {
  if (typeof window === "undefined") return;

  const linkId = `font-${fontName.replace(/\s+/g, "-").toLowerCase()}`;
  if (!document.getElementById(linkId)) {
    const link = document.createElement("link");
    link.id = linkId;
    link.rel = "stylesheet";
    link.href = `https://fonts.googleapis.com/css2?family=${fontName.replace(
      /\s+/g,
      "+"
    )}:wght@300;400;500;600;700;800&display=swap`;
    document.head.appendChild(link);
  }
}

/**
 * Apply all brand kit styles dynamically
 */
export function applyBrandKitStyles(
  typographySettings: TypographySettings | null,
  colorSettings: any | null
): void {
  if (typeof window === "undefined") return;

  let styles = "";

  // Get Typography values
  let headingsFamily = "Inter";
  let headingsWeight = "700";
  let bodyFamily = "Inter";
  let bodySize = 16;
  let bodyWeight = "400";

  // Apply Typography Styles
  if (typographySettings) {
    headingsFamily =
      typographySettings.headings?.family ||
      typographySettings.headings_family ||
      "Inter";
    headingsWeight =
      typographySettings.headings?.weight ||
      typographySettings.headings_weight ||
      "700";
    bodyFamily =
      typographySettings.body?.family ||
      typographySettings.body_family ||
      "Inter";
    bodySize =
      typographySettings.body?.baseSize ||
      typographySettings.body_base_size ||
      16;
    bodyWeight =
      typographySettings.body?.weight ||
      typographySettings.body_weight ||
      "400";

    styles += `
      .m-page h1, .m-page h2, .m-page h3, .m-page h4, .m-page h5, .m-page h6 {
        font-family: '${headingsFamily}', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif !important;
        font-weight: ${headingsWeight} !important;
      }
      .m-page, .m-page p, .m-page span, .m-page div, .m-page a, .m-page button {
        font-family: '${bodyFamily}', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif !important;
        font-size: ${bodySize}px !important;
        font-weight: ${bodyWeight} !important;
      }
    `;
  }

  // Apply Color Styles
  if (colorSettings) {
    const bgColor = colorSettings.background || "#ffffff";
    const borderColor = colorSettings.border || "#e2e8f0";
    const isBorder = colorSettings.is_border !== false;
    const shadowColor = colorSettings.shadow || "#000000";
    const isShadow = colorSettings.is_shadow !== false;
    const primaryText = colorSettings.primary_text || "#1e293b";
    const secondaryText = colorSettings.secondary_text || "#64748b";
    const buttonText = colorSettings.apply_button_text || "#ffffff";
    const buttonBg = colorSettings.apply_button_background || "#006980";

    styles += `
      .m-page .card-list {
        background-color: ${bgColor} !important;
        ${
          isBorder
            ? `border: 1px solid ${borderColor} !important;`
            : "border: none !important;"
        }
        ${
          isShadow
            ? `box-shadow: 0px 1px 1px 0px ${shadowColor}25 !important;`
            : "box-shadow: none !important;"
        }
      }
      .m-page .name-header,
      .m-page h1, .m-page h2, .m-page h3, .m-page h4, .m-page h5, .m-page h6 {
        color: ${primaryText} !important;
      }
      .m-page .gray-6,
      .m-page .company-description,
      .m-page .company-info-badge,
      .m-page .company-info-badge span,
      .m-page p {
        color: ${secondaryText} !important;
      }
      .m-page .btn-theme {
        background-color: ${buttonBg} !important;
        color: ${buttonText} !important;
      }
      .m-page .color-p,
      .m-page .color-p i,
      .m-page .color-p span {
        color: ${buttonBg} !important;
      }
      .m-page .btn-theme-outline {
        border-color: ${buttonBg} !important;
        color: ${buttonBg} !important;
      }
      .m-page .btn-theme-outline:hover {
        background-color: ${buttonBg} !important;
        color: ${buttonText} !important;
      }
      .m-page .flex.justify-center.items-center a,
      .m-page .flex.justify-center.items-center span {
        color: ${buttonBg} !important;
        font-family: '${bodyFamily}', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif !important;
      }
      .m-page .flex.justify-center.items-center a:hover {
        opacity: 0.8;
      }
      .m-page .flex.justify-center.items-center i {
        color: ${buttonBg} !important;
      }
      .m-page .flex.justify-center.items-center {
        font-family: '${bodyFamily}', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif !important;
      }
    `;
  }

  // Inject styles
  console.log("🎨 [Brand Kit] Generated CSS styles length:", styles.length);
  console.log("🎨 [Brand Kit] CSS preview:", styles.substring(0, 200));
  injectStyles(styles);
}

/**
 * Inject CSS styles into the document
 */
function injectStyles(css: string): void {
  if (typeof window === "undefined") return;

  // Remove existing brand kit styles
  const existingStyle = document.getElementById("brand-kit-styles");
  if (existingStyle) {
    try {
      // Safely check if element is still in the DOM before removing
      if (existingStyle.parentNode && document.contains(existingStyle)) {
        existingStyle.remove();
        console.log("🗑️ [Brand Kit] Removed existing styles");
      }
    } catch (error) {
      console.warn("Error removing brand kit styles:", error);
    }
  }

  // Add new styles
  const style = document.createElement("style");
  style.id = "brand-kit-styles";
  style.textContent = css;
  document.head.appendChild(style);
  console.log(
    "✅ [Brand Kit] CSS injected into document.head, style element ID:",
    style.id
  );
}

/**
 * Get logo URL for header/footer
 */
export function getLogoUrl(
  logoSettings: LogoSettings | null,
  profileLogo: string | null,
  type: "header" | "footer" = "header"
): string {
  if (!logoSettings) {
    return profileLogo || "";
  }

  // Handle nested format
  if (logoSettings[type]) {
    const logo = logoSettings[type];
    if (logo?.type === "text" && logo.text) {
      return ""; // Text logo, return empty to show text
    }
    const imageUrl = logo?.imageUrl || logo?.image_url;
    if (imageUrl) {
      return imageUrl.startsWith("http") ? imageUrl : imageUrl;
    }
  }

  // Handle flat snake_case format
  const logoUrlKey = type === "header" ? "header_logo_url" : "footer_logo_url";
  const logoTypeKey =
    type === "header" ? "header_logo_type" : "footer_logo_type";

  if (
    logoSettings[logoTypeKey as keyof LogoSettings] === "text" &&
    logoSettings[
      (type === "header" ? "header_text" : "footer_text") as keyof LogoSettings
    ]
  ) {
    return ""; // Text logo
  }

  const imageUrl = logoSettings[logoUrlKey as keyof LogoSettings] as
    | string
    | undefined;
  if (imageUrl) {
    return imageUrl.startsWith("http") ? imageUrl : imageUrl;
  }

  // Fallback to profile logo
  return profileLogo || "";
}

/**
 * Check if logo is text-based
 */
export function isTextLogo(
  logoSettings: LogoSettings | null,
  type: "header" | "footer" = "header"
): boolean {
  if (!logoSettings) return false;

  // Handle nested format
  if (logoSettings[type]) {
    return logoSettings[type]?.type === "text";
  }

  // Handle flat format
  const logoTypeKey =
    type === "header" ? "header_logo_type" : "footer_logo_type";
  return logoSettings[logoTypeKey as keyof LogoSettings] === "text";
}

/**
 * Get text logo content
 */
export function getTextLogoContent(
  logoSettings: LogoSettings | null,
  type: "header" | "footer" = "header"
): {
  text: string;
  fontFamily: string;
  fontSize: number;
  color: string;
  linkUrl: string;
  linkTarget: string;
} | null {
  if (!logoSettings) return null;

  // Handle nested format
  if (logoSettings[type] && logoSettings[type]?.type === "text") {
    const logo = logoSettings[type];
    return {
      text: logo?.text || "",
      fontFamily: logo?.fontFamily || logo?.font_family || "Inter",
      fontSize: logo?.fontSize || logo?.font_size || 24,
      color: logo?.color || "#006980",
      linkUrl: logo?.linkUrl || logo?.link_url || "",
      linkTarget: logo?.linkTarget || logo?.link_target || "_self",
    };
  }

  // Handle flat format
  const logoTypeKey =
    type === "header" ? "header_logo_type" : "footer_logo_type";
  if (logoSettings[logoTypeKey as keyof LogoSettings] === "text") {
    return {
      text:
        (logoSettings[
          (type === "header"
            ? "header_text"
            : "footer_text") as keyof LogoSettings
        ] as string) || "",
      fontFamily:
        (logoSettings[
          (type === "header"
            ? "header_font_family"
            : "footer_font_family") as keyof LogoSettings
        ] as string) || "Inter",
      fontSize:
        (logoSettings[
          (type === "header"
            ? "header_font_size"
            : "footer_font_size") as keyof LogoSettings
        ] as number) || 24,
      color:
        (logoSettings[
          (type === "header"
            ? "header_color"
            : "footer_color") as keyof LogoSettings
        ] as string) || "#006980",
      linkUrl:
        (logoSettings[
          (type === "header"
            ? "header_link_url"
            : "footer_link_url") as keyof LogoSettings
        ] as string) || "",
      linkTarget:
        (logoSettings[
          (type === "header"
            ? "header_link_target"
            : "footer_link_target") as keyof LogoSettings
        ] as string) || "_self",
    };
  }

  return null;
}

/**
 * Check if footer link should be shown
 */
export function shouldShowFooterLink(
  footerSettings: FooterSettings | null,
  key: string
): boolean {
  // If no footer settings exist, show all links by default (backward compatibility)
  if (!footerSettings) {
    console.log(
      `🔗 [Footer] shouldShowFooterLink("${key}") - No settings, returning true`
    );
    return true;
  }

  // Backend returns snake_case, so try snake_case first
  const snakeCaseKey = key.replace(/([A-Z])/g, "_$1").toLowerCase();

  // Try snake_case first (backend format)
  let value = footerSettings[snakeCaseKey as keyof FooterSettings];

  // If not found, try camelCase version (fallback)
  if (value === undefined || value === null) {
    value = footerSettings[key as keyof FooterSettings];
  }

  // If value is explicitly set, use it
  if (value !== undefined && value !== null) {
    let result: boolean = false;
    if (typeof value === "boolean") {
      result = value === true;
    } else if (typeof value === "number") {
      result = value === 1;
    } else if (typeof value === "string") {
      const lowerValue = value.toLowerCase().trim();
      result =
        lowerValue === "true" || lowerValue === "1" || lowerValue === "yes";
    } else {
      result = Boolean(value);
    }
    console.log(
      `🔗 [Footer] shouldShowFooterLink("${key}") - Value: ${value} (${typeof value}), Result: ${result}`
    );
    return result;
  }

  // Default: show if not specified
  console.log(
    `🔗 [Footer] shouldShowFooterLink("${key}") - No value found, returning true (default)`
  );
  return true;
}

/**
 * Get platform data for social links
 */
export function getPlatformData(platformId: string): {
  id: string;
  name: string;
  icon: string;
  color: string;
} {
  const platforms: Record<
    string,
    { id: string; name: string; icon: string; color: string }
  > = {
    facebook: {
      id: "facebook",
      name: "Facebook",
      icon: "fa-brands fa-facebook",
      color: "#1877F2",
    },
    twitter: {
      id: "twitter",
      name: "Twitter / X",
      icon: "fa-brands fa-twitter",
      color: "#1DA1F2",
    },
    linkedin: {
      id: "linkedin",
      name: "LinkedIn",
      icon: "fa-brands fa-linkedin",
      color: "#0A66C2",
    },
    instagram: {
      id: "instagram",
      name: "Instagram",
      icon: "fa-brands fa-instagram",
      color: "#E4405F",
    },
    youtube: {
      id: "youtube",
      name: "YouTube",
      icon: "fa-brands fa-youtube",
      color: "#FF0000",
    },
    tiktok: {
      id: "tiktok",
      name: "TikTok",
      icon: "fa-brands fa-tiktok",
      color: "#000000",
    },
    github: {
      id: "github",
      name: "GitHub",
      icon: "fa-brands fa-github",
      color: "#181717",
    },
    whatsapp: {
      id: "whatsapp",
      name: "WhatsApp",
      icon: "fa-brands fa-whatsapp",
      color: "#25D366",
    },
    custom: {
      id: "custom",
      name: "Custom Link",
      icon: "link",
      color: "#64748B",
    },
  };

  if (platformId === "custom") {
    return { id: "custom", name: "Custom", icon: "link", color: "#64748B" };
  }
  return platforms[platformId] || platforms.custom;
}

/**
 * Get icon for social link
 */
export function getSocialLinkIcon(link: SocialLink): string {
  const platform = link.platform || link.platform_id;

  if (platform === "custom") {
    const customIcon = link.customIcon || link.custom_icon;
    if (customIcon && customIcon.trim()) {
      return customIcon.trim();
    }
    return "link";
  }

  if (platform) {
    const platformData = getPlatformData(platform);
    if (platformData && platformData.icon) {
      return platformData.icon;
    }
  }

  return "link";
}

/**
 * Get display name for social link
 */
export function getSocialLinkName(link: SocialLink): string {
  const platform = link.platform || link.platform_id;

  if (platform === "custom") {
    const customName = link.customName || link.custom_name;
    if (customName && customName.trim()) {
      return customName.trim();
    }
    return "Custom Link";
  }

  if (platform) {
    const platformData = getPlatformData(platform);
    if (platformData && platformData.name) {
      return platformData.name;
    }
  }

  return "Link";
}

/**
 * Check if icon is FontAwesome
 */
export function isFontAwesome(icon: string): boolean {
  return !!(icon && icon.includes("fa-"));
}

/**
 * Get styles for social link
 */
export function getSocialLinkStyles(
  link: SocialLink,
  socialStyling: SocialStyling | null
): React.CSSProperties {
  if (!socialStyling) {
    return {};
  }

  const platform = link.platform || link.platform_id;
  const platformData = platform ? getPlatformData(platform) : null;

  const shape = socialStyling.shape || "circle";
  const size = socialStyling.size || 40;
  const colorMode =
    socialStyling.colorMode || socialStyling.color_mode || "platform";
  const customColor =
    socialStyling.customColor || socialStyling.custom_color || "#006980";
  const background = socialStyling.background || "none";
  const customBgColor =
    socialStyling.customBgColor || socialStyling.custom_bg_color || "#006980";

  // Base styles
  const styles: React.CSSProperties = {
    width: `${size}px`,
    height: `${size}px`,
    fontSize: `${size * 0.6}px`,
    display: "flex",
    alignItems: "center",
    justifyContent: "center",
    minWidth: `${size}px`,
    minHeight: `${size}px`,
  };

  // Shape
  if (shape === "circle") {
    styles.borderRadius = "50%";
  } else if (shape === "rounded") {
    styles.borderRadius = "8px";
  } else {
    styles.borderRadius = "0";
  }

  // Icon Colors
  let iconColor = "#000000";

  if (colorMode === "platform" && platformData) {
    iconColor = platformData.color;
  } else if (colorMode === "mono") {
    iconColor = "#000000";
  } else if (colorMode === "white") {
    iconColor = "#FFFFFF";
  } else if (colorMode === "grey") {
    iconColor = "#64748B";
  } else if (colorMode === "custom") {
    iconColor = customColor;
  }

  styles.color = iconColor;

  // Background Styles
  if (background === "filled") {
    if (colorMode === "platform" && platformData) {
      styles.backgroundColor = platformData.color;
    } else if (colorMode === "mono") {
      styles.backgroundColor = "#000000";
    } else if (colorMode === "white") {
      styles.backgroundColor = "#FFFFFF";
    } else if (colorMode === "grey") {
      styles.backgroundColor = "#64748B";
    } else if (colorMode === "custom") {
      styles.backgroundColor = iconColor;
    }
    styles.padding = "8px";
  } else if (background === "outlined") {
    styles.border = `2px solid ${iconColor}`;
    styles.borderColor = iconColor;
    styles.backgroundColor = "transparent";
    styles.padding = "8px";
    styles.boxSizing = "border-box";
  } else if (background === "custom") {
    styles.backgroundColor = customBgColor || "#006980";
    styles.padding = "8px";
  } else {
    styles.backgroundColor = "transparent";
    styles.border = "none";
    styles.padding = "0";
  }

  return styles;
}

/**
 * Get icon color style
 */
export function getIconColorStyle(
  link: SocialLink,
  socialStyling: SocialStyling | null
): React.CSSProperties {
  if (!socialStyling) {
    return { color: "inherit" };
  }

  const platform = link.platform || link.platform_id;
  const platformData = platform ? getPlatformData(platform) : null;
  const background = socialStyling.background || "none";
  const colorMode =
    socialStyling.colorMode || socialStyling.color_mode || "platform";
  const customColor =
    socialStyling.customColor || socialStyling.custom_color || "#006980";

  let iconColor = "#000000";

  if (background === "filled") {
    if (colorMode === "white") {
      iconColor = "#000000";
    } else {
      iconColor = "#FFFFFF";
    }
  } else if (background === "outlined") {
    if (colorMode === "platform" && platformData) {
      iconColor = platformData.color;
    } else if (colorMode === "mono") {
      iconColor = "#000000";
    } else if (colorMode === "white") {
      iconColor = "#FFFFFF";
    } else if (colorMode === "grey") {
      iconColor = "#64748B";
    } else if (colorMode === "custom") {
      iconColor = customColor;
    }
  } else if (background === "custom") {
    iconColor = "#FFFFFF";
  } else {
    if (colorMode === "platform" && platformData) {
      iconColor = platformData.color;
    } else if (colorMode === "mono") {
      iconColor = "#000000";
    } else if (colorMode === "white") {
      iconColor = "#FFFFFF";
    } else if (colorMode === "grey") {
      iconColor = "#64748B";
    } else if (colorMode === "custom") {
      iconColor = customColor;
    }
  }

  return { color: iconColor };
}

/**
 * Get CSS class for social link hover effect
 */
export function getSocialLinkClass(
  socialStyling: SocialStyling | null
): string {
  if (!socialStyling) {
    return "";
  }
  const hoverEffect =
    socialStyling.hoverEffect || socialStyling.hover_effect || "lift";
  return `social-link-hover-${hoverEffect}`;
}

/**
 * Apply favicon dynamically
 */
export function applyFavicon(
  logoSettings: LogoSettings | null,
  profileLogo: string | null
): void {
  if (typeof window === "undefined") return;

  try {
    let faviconUrl: string | null = null;

    if (logoSettings) {
      // Handle nested format
      if (logoSettings.favicon) {
        const favicon = logoSettings.favicon;
        const imageUrl = favicon.imageUrl || favicon.image_url;
        if (imageUrl) {
          faviconUrl = imageUrl.startsWith("http") ? imageUrl : imageUrl;
        }
      }
      // Handle flat snake_case format
      else if (logoSettings.favicon_url) {
        const imageUrl = logoSettings.favicon_url;
        faviconUrl = imageUrl.startsWith("http") ? imageUrl : imageUrl;
      }
    }

    // Fallback to profile logo
    if (!faviconUrl && profileLogo) {
      faviconUrl = profileLogo.startsWith("http") ? profileLogo : profileLogo;
    }

    // Remove existing favicon links
    const existingFavicons = document.querySelectorAll(
      'link[rel="icon"], link[rel="shortcut icon"], link[rel="apple-touch-icon"]'
    );
    // Convert NodeList to Array to avoid issues during iteration
    Array.from(existingFavicons).forEach((link) => {
      if (link && link.parentNode && link.parentNode === document.head) {
        try {
          link.remove();
        } catch (error) {
          // Ignore errors - element may have been removed already
        }
      }
    });

    if (faviconUrl) {
      // Add standard favicon
      const faviconLink = document.createElement("link");
      faviconLink.setAttribute("rel", "icon");
      faviconLink.setAttribute("type", "image/png");
      faviconLink.setAttribute("href", faviconUrl);
      document.head.appendChild(faviconLink);

      // Add shortcut icon for older browsers
      const shortcutLink = document.createElement("link");
      shortcutLink.setAttribute("rel", "shortcut icon");
      shortcutLink.setAttribute("href", faviconUrl);
      document.head.appendChild(shortcutLink);

      // Add Apple touch icon
      const appleTouchIcon = document.createElement("link");
      appleTouchIcon.setAttribute("rel", "apple-touch-icon");
      appleTouchIcon.setAttribute("href", faviconUrl);
      document.head.appendChild(appleTouchIcon);
    }
  } catch (error) {
    console.error("Error applying favicon:", error);
  }
}
