// API Configuration and utilities
// Backend base URL - matches company-dashboard-flowxtra-node/config/config.js
// Production: https://app.flowxtra.com
// Development: http://127.0.0.1:8000

// Determine backend URL based on environment
const getBackendBaseUrl = (): string => {
  // If explicitly set via environment variable, use it
  if (process.env.NEXT_PUBLIC_API_BASE_URL) {
    return process.env.NEXT_PUBLIC_API_BASE_URL;
  }

  // Production: use backend server
  if (process.env.NODE_ENV === "production") {
    return "https://app.flowxtra.com";
  }

  // Development: use local server
  return "http://127.0.0.1:8000";
};

const BACKEND_BASE_URL = getBackendBaseUrl();

// Full API base URL with /api path
export const API_BASE_URL = `${BACKEND_BASE_URL}/api`;

export const API_ENDPOINTS = {
  Show: "/companies/show_share",
  CreateProfilFilter: "/companies/createProfilFilter",
  JobShow: "/jobs/show",
  JobApplyQuestions: "/jobs/apply/questions",
  JobApply: "/candidate-job/store", // Matches Angular API.Store
  JobApplyAfterLogin: "/candidate-job/applyAfterLogin", // Matches Angular API.applyAfterLogin
  UserSearch: "/users/search", // Matches Angular API.Search
  LoginApply: "/auth/login_apply", // Matches Angular API.loginApply
  JobCompanySitemap: "/job-company-sitemap",
  JobCompanyFeed: "/job-company-feed",
  GetAllPolicies: "/company-policies/index", // Matches Angular API.GetAllPolicies
  GetPolicyBySlug: "/companies/policies", // Public endpoint for getting policy by slug
};

export interface ApiResponse<T> {
  success: boolean;
  data: T;
  message?: string;
}

export interface PaginateData<T> {
  data: T[];
  current_page: number;
  last_page: number;
  per_page: number;
  total: number;
}

export async function apiGet<T>(
  endpoint: string,
  params?: Record<string, any>,
  subdomain?: string
): Promise<ApiResponse<T>> {
  try {
    const url = new URL(`${API_BASE_URL}${endpoint}`);

    // Add subdomain as query parameter if provided
    if (subdomain && subdomain !== "localhost" && subdomain !== "127.0.0.1") {
      url.searchParams.append("subdomain", subdomain);
    }

    if (params) {
      Object.keys(params).forEach((key) => {
        if (
          params[key] !== null &&
          params[key] !== undefined &&
          params[key] !== ""
        ) {
          url.searchParams.append(key, params[key].toString());
        }
      });
    }

    const headers: HeadersInit = {
      "Content-Type": "application/json",
    };

    // Get token from localStorage if available (client-side only)
    let token: string | null = null;
    if (typeof window !== "undefined") {
      // Matches Angular TOKEN_KEY = "flowxtra-token"
      token = localStorage.getItem("flowxtra-token") || null;
    }

    // Add Authorization header if token exists
    if (token && token.trim() !== "") {
      headers["Authorization"] = `Bearer ${token}`;
    }

    const response = await fetch(url.toString(), {
      method: "GET",
      headers,
      cache: "no-store", // Ensure fresh data
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.message || `API Error: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error("API Error:", error);
    throw error;
  }
}

export async function apiPost<T>(
  endpoint: string,
  data: FormData | Record<string, any>,
  subdomain?: string
): Promise<ApiResponse<T>> {
  try {
    const url = new URL(`${API_BASE_URL}${endpoint}`);

    // Add subdomain as query parameter if provided
    if (subdomain && subdomain !== "localhost" && subdomain !== "127.0.0.1") {
      url.searchParams.append("subdomain", subdomain);
    }

    const headers: HeadersInit = {};

    // Get token from localStorage if available (client-side only)
    let token: string | null = null;
    if (typeof window !== "undefined") {
      // Matches Angular TOKEN_KEY = "flowxtra-token"
      token = localStorage.getItem("flowxtra-token") || null;
    }

    // Add Authorization header if token exists
    if (token && token.trim() !== "") {
      headers["Authorization"] = `Bearer ${token}`;
    }

    // If FormData, don't set Content-Type (browser will set it with boundary)
    if (!(data instanceof FormData)) {
      headers["Content-Type"] = "application/json";
    }

    const body = data instanceof FormData ? data : JSON.stringify(data);

    const response = await fetch(url.toString(), {
      method: "POST",
      headers,
      body,
    });

    if (!response.ok) {
      let errorData: any = {};
      try {
        errorData = await response.json();
      } catch {
        errorData = { message: `API Error: ${response.status}` };
      }

      // Create error object with message and status
      const error = new Error(
        errorData.message || errorData.error || `API Error: ${response.status}`
      );
      (error as any).status = response.status;
      (error as any).data = errorData;
      throw error;
    }

    return await response.json();
  } catch (error) {
    console.error("API Error:", error);
    throw error;
  }
}
