import { NextRequest, NextResponse } from "next/server";
import xmlbuilder from "xmlbuilder";
import { API_BASE_URL, API_ENDPOINTS } from "@/lib/api";
import {
  getOriginalHost,
  getProto,
  getSubdomainFromHost,
  absolutize,
  isoNoMillis,
  sha1,
} from "@/lib/utils-server";

function buildXml(
  jobRoutes: any[],
  host: string,
  proto: string,
  companyId: string
): string {
  const root = xmlbuilder
    .create("urlset", { version: "1.0" })
    .att("xmlns", "http://www.sitemaps.org/schemas/sitemap/0.9");

  if (Array.isArray(jobRoutes) && jobRoutes.length) {
    for (const r of jobRoutes) {
      const loc = typeof r === "string" ? r : r.loc;
      const lastmod =
        typeof r === "object" && r.lastmod
          ? r.lastmod.match(/Z$/)
            ? r.lastmod
            : isoNoMillis(new Date(r.lastmod))
          : isoNoMillis(new Date());
      const changefreq =
        typeof r === "object" && r.changefreq ? r.changefreq : "daily";
      const priority =
        typeof r === "object" && r.priority ? r.priority : "0.8";

      const url = root.ele("url");
      url.ele("loc", absolutize(loc, host, proto));
      url.ele("lastmod", lastmod);
      url.ele("changefreq", changefreq);
      url.ele("priority", priority);
    }
  } else {
    const url = root.ele("url");
    url.ele("loc", `${proto}://${host}/company/${companyId}`);
    url.ele("lastmod", isoNoMillis(new Date()));
    url.ele("changefreq", "daily");
    url.ele("priority", "1.0");
  }

  return root.end({ pretty: true, indent: "  " });
}

export async function GET(request: NextRequest) {
  try {
    const headers = request.headers;
    const host = getOriginalHost(headers);
    const subdomain = getSubdomainFromHost(host) || "default";
    const proto = getProto(headers);

    const apiUrl = `${API_BASE_URL}${API_ENDPOINTS.JobCompanySitemap}?subdomain=${subdomain}`;

    const response = await fetch(apiUrl, {
      headers: {
        "User-Agent": "Flowxtra-Sitemap-Generator/1.0",
        Accept: "application/json",
        Authorization: headers.get("authorization") || "",
      },
      cache: "no-store",
    });

    let list: any[] = [];

    if (response.ok) {
      const data = await response.json();
      list = Array.isArray(data) ? data : [];
    } else {
      console.error("Backend API Error:", response.status);
    }

    const xml = buildXml(list, host, proto, subdomain);

    // Headers + caching
    const etag = `W/"${sha1(xml)}"`;
    const ifNoneMatch = headers.get("if-none-match");

    if (ifNoneMatch && ifNoneMatch === etag) {
      return new NextResponse(null, { status: 304 });
    }

    return new NextResponse(xml.trim(), {
      status: 200,
      headers: {
        "Content-Type": "application/xml; charset=utf-8",
        "Cache-Control": "public, max-age=3600, s-maxage=3600",
        "X-Robots-Tag": "all",
        "Access-Control-Allow-Origin": "*",
        "Access-Control-Allow-Methods": "GET",
        "Access-Control-Allow-Headers": "Content-Type",
        ETag: etag,
        "Last-Modified": new Date().toUTCString(),
      },
    });
  } catch (error) {
    console.error("Error generating sitemap:", error);
    const headers = request.headers;
    const host = getOriginalHost(headers);
    const proto = getProto(headers);
    const subdomain = getSubdomainFromHost(host) || "default";
    const xml = buildXml([], host, proto, subdomain);

    return new NextResponse(xml.trim(), {
      status: 200,
      headers: {
        "Content-Type": "application/xml; charset=utf-8",
        "Cache-Control": "public, max-age=3600, s-maxage=3600",
      },
    });
  }
}

