import { NextRequest, NextResponse } from "next/server";
import { API_BASE_URL, API_ENDPOINTS } from "@/lib/api";
import {
  getOriginalHost,
  getSubdomainFromHost,
} from "@/lib/utils-server";

async function getCompanyInfo(subdomain: string) {
  try {
    const url = `${API_BASE_URL}${API_ENDPOINTS.Show}/${subdomain}`;
    const response = await fetch(url, {
      cache: "no-store",
    });
    if (response.ok) {
      return await response.json();
    }
    return null;
  } catch (error) {
    console.error("Error fetching company info:", error);
    return null;
  }
}

function generateRobotsTxt({ sitemapUrl }: { sitemapUrl: string }): string {
  return `User-agent: *
Allow: /

Sitemap: ${sitemapUrl}`;
}

export async function GET(request: NextRequest) {
  try {
    const headers = request.headers;
    const host = getOriginalHost(headers);
    const subdomain = getSubdomainFromHost(host) || "default";

    const companyInfo = await getCompanyInfo(subdomain);

    const companyData = companyInfo?.data || {};
    const profile = companyData.profile || {};
    const id = profile.id;
    const companyDomain = companyInfo?.domain || `${subdomain}.flowxtra.com`;
    const sitemapFromApi = companyInfo?.sitemap_url;

    let sitemapUrl = sitemapFromApi ||
      (id
        ? `https://${companyDomain}/${id}/sitemap.xml`
        : `https://${companyDomain}/sitemap.xml`);

    const robotsContent = generateRobotsTxt({ sitemapUrl });

    return new NextResponse(robotsContent, {
      status: 200,
      headers: {
        "Content-Type": "text/plain",
      },
    });
  } catch (error) {
    console.error("Error generating robots.txt:", error);
    const headers = request.headers;
    const host = getOriginalHost(headers);
    const subdomain = getSubdomainFromHost(host) || "default";
    const sitemapUrl = `https://${subdomain}.flowxtra.com/sitemap.xml`;
    const robotsContent = generateRobotsTxt({ sitemapUrl });

    return new NextResponse(robotsContent, {
      status: 200,
      headers: {
        "Content-Type": "text/plain",
      },
    });
  }
}

