"use client";

import { useEffect, useState } from "react";
import { apiGet, API_ENDPOINTS } from "@/lib/api";
import type {
  Company,
  CompanyShowSocial,
  Job,
  FilterData,
  ProfileData,
  FilterObj,
} from "@/lib/types";
import { sanitizeTitle, formatSalary, getSubdomain, getUrlSitemap } from "@/lib/utils";
import Image from "next/image";
import Link from "next/link";
import { useRouter } from "next/navigation";
import MetaTags from "@/app/components/MetaTags";
import type {
  TypographySettings,
  LogoSettings,
  FooterSettings,
  SocialLink,
  SocialStyling,
} from "@/lib/types";
import {
  loadTypographyFonts,
  applyBrandKitStyles,
  getLogoUrl,
  isTextLogo,
  getTextLogoContent,
  shouldShowFooterLink,
  getSocialLinkIcon,
  getSocialLinkName,
  isFontAwesome,
  getSocialLinkStyles,
  getIconColorStyle,
  getSocialLinkClass,
  applyFavicon,
} from "@/lib/brand-kit-utils";
import ReportIssueModal, { ReportIssueData } from "@/app/components/ReportIssueModal";

export default function ProfilePage() {
  const router = useRouter();
  const [isLoading, setIsLoading] = useState(false);
  const [profile, setProfile] = useState<Company | null>(null);
  const [jobs, setJobs] = useState<Job[]>([]);
  const [companyImageGallery, setCompanyImageGallery] = useState<
    Array<{ imge: string }>
  >([]);
  const [companyShowSocial, setCompanyShowSocial] = useState<CompanyShowSocial | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [filterObj, setFilterObj] = useState<FilterObj>({});
  const [filterData, setFilterData] = useState<FilterData | null>(null);
  const [subdomain, setSubdomain] = useState("");

  // Brand Kit Settings
  const [typographySettings, setTypographySettings] = useState<TypographySettings | null>(null);
  const [logoSettings, setLogoSettings] = useState<LogoSettings | null>(null);
  const [footerSettings, setFooterSettings] = useState<FooterSettings | null>(null);
  const [socialLinks, setSocialLinks] = useState<SocialLink[]>([]);
  const [socialStyling, setSocialStyling] = useState<SocialStyling | null>(null);
  const [isReportModalOpen, setIsReportModalOpen] = useState(false);
  const [publishedPolicies, setPublishedPolicies] = useState<any[]>([]);

  useEffect(() => {
    if (typeof window !== "undefined") {
      window.scrollTo(0, 0);
      const sub = getSubdomain();
      setSubdomain(sub);

      if (sub === "my") {
        router.push("/login");
        return;
      }

      // Load data with subdomain directly
      const loadData = async () => {
        try {
          // Load filter data first
          const filterResponse = await apiGet<FilterData>(
            API_ENDPOINTS.CreateProfilFilter,
            undefined,
            sub
          );
          setFilterData(filterResponse.data);

          // Then load profile
          const params: Record<string, any> = {
            page: "1",
            max: "10",
          };
          const profileResponse = await apiGet<ProfileData>(
            API_ENDPOINTS.Show,
            params,
            sub
          );
          setProfile(profileResponse.data.profile);
          setJobs(profileResponse.data.jobs.data);
          setCurrentPage(profileResponse.data.jobs.current_page);
          setTotalPages(profileResponse.data.jobs.last_page);

          if (profileResponse.data.profile?.company_image_gallery) {
            setCompanyImageGallery(profileResponse.data.profile.company_image_gallery);
          } else {
            setCompanyImageGallery([]);
          }

          if (profileResponse.data.profile?.company_show_social) {
            setCompanyShowSocial(profileResponse.data.profile.company_show_social);
          } else {
            setCompanyShowSocial(null);
          }

          // Load Brand Kit Settings
          const profileData = profileResponse.data.profile as any;
          console.log("🔍 [Brand Kit] API Response - profileData:", profileData);
          console.log("🔍 [Brand Kit] Typography Settings:", profileData.typography_settings);
          console.log("🔍 [Brand Kit] Logo Settings:", profileData.logo_settings || profileData.logo_setting);
          console.log("🔍 [Brand Kit] Footer Settings:", profileData.footer_settings);
          console.log("🔍 [Brand Kit] Social Links:", profileData.social_links);
          console.log("🔍 [Brand Kit] Social Styling:", profileData.social_styling);
          console.log("🔍 [Brand Kit] Company Show Social:", profileData.company_show_social);
          
          if (profileData.typography_settings) {
            setTypographySettings(profileData.typography_settings);
            console.log("✅ [Brand Kit] Typography Settings loaded");
          }
          if (profileData.logo_settings || profileData.logo_setting) {
            setLogoSettings(profileData.logo_settings || profileData.logo_setting);
            console.log("✅ [Brand Kit] Logo Settings loaded");
          }
          if (profileData.footer_settings) {
            setFooterSettings(profileData.footer_settings);
            console.log("✅ [Brand Kit] Footer Settings loaded:", profileData.footer_settings);
          } else {
            console.log("⚠️ [Brand Kit] No Footer Settings found");
          }
          if (profileData.social_links && Array.isArray(profileData.social_links)) {
            setSocialLinks(profileData.social_links);
            console.log("✅ [Brand Kit] Social Links loaded:", profileData.social_links.length, "links");
          } else {
            console.log("⚠️ [Brand Kit] No Social Links found");
          }
          if (profileData.social_styling) {
            setSocialStyling(profileData.social_styling);
            console.log("✅ [Brand Kit] Social Styling loaded:", profileData.social_styling);
          } else {
            console.log("⚠️ [Brand Kit] No Social Styling found");
          }

          setIsLoading(false);
        } catch (error) {
          console.error("Error loading data:", error);
          setIsLoading(false);
        }
      };

      loadData();
    }
  }, [router]);

  useEffect(() => {
    if (filterData && subdomain) {
      loadProfile(1);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [filterObj]);

  // Apply Brand Kit Styles
  useEffect(() => {
    if (typeof window === "undefined") return;
    
    console.log("🎨 [Brand Kit] Applying styles - Typography:", typographySettings);
    console.log("🎨 [Brand Kit] Applying styles - Colors:", companyShowSocial);
    
    // Small delay to ensure DOM is ready
    const timeoutId = setTimeout(() => {
      if (typographySettings) {
        loadTypographyFonts(typographySettings);
        console.log("✅ [Brand Kit] Typography fonts loaded");
      }
      
      applyBrandKitStyles(typographySettings, companyShowSocial);
      console.log("✅ [Brand Kit] Styles applied");
      
      // Verify CSS injection
      const styleElement = document.getElementById("brand-kit-styles");
      if (styleElement) {
        console.log("✅ [Brand Kit] CSS injected successfully");
        console.log("🎨 [Brand Kit] CSS content preview:", styleElement.textContent?.substring(0, 300));
        
        // Verify selectors match elements
        const cardListElements = document.querySelectorAll(".m-page .card-list");
        const btnThemeElements = document.querySelectorAll(".m-page .btn-theme");
        const nameHeaderElements = document.querySelectorAll(".m-page .name-header");
        const gray6Elements = document.querySelectorAll(".m-page .gray-6");
        console.log("🎨 [Brand Kit] Found elements - card-list:", cardListElements.length, "btn-theme:", btnThemeElements.length, "name-header:", nameHeaderElements.length, "gray-6:", gray6Elements.length);
      } else {
        console.log("⚠️ [Brand Kit] CSS not injected - style element not found");
      }
    }, 100);
    
    return () => clearTimeout(timeoutId);
  }, [typographySettings, companyShowSocial]);

  // Apply Favicon
  useEffect(() => {
    if (logoSettings || profile?.logo) {
      applyFavicon(logoSettings, profile?.logo || null);
    }
  }, [logoSettings, profile?.logo]);

  // Open Report Issue Modal
  const openReportIssue = () => {
    setIsReportModalOpen(true);
  };

  // Handle Report Modal Close
  const handleReportModalClose = (result?: { success: boolean; data?: any; error?: any }) => {
    setIsReportModalOpen(false);
    if (result?.success) {
      // Show success message (you can use toast notification here)
      console.log("Report submitted successfully:", result.data);
    } else if (result?.error) {
      // Show error message (you can use toast notification here)
      console.error("Failed to submit report:", result.error);
    }
  };

  // Load published policies
  const loadPublishedPolicies = async (subdomain: string) => {
    try {
      // Use the same API endpoint as Dashboard: /company-policies/index
      // This endpoint requires authentication, but for public pages we use the public endpoint
      // Try public endpoint first, fallback to authenticated if token exists
      let response;
      try {
        // Try public endpoint first (for published policies only)
        response = await apiGet<any[]>(
          API_ENDPOINTS.GetPolicyBySlug,
          {},
          subdomain
        );
      } catch (publicError) {
        // If public endpoint fails, try authenticated endpoint if token exists
        if (typeof window !== "undefined" && localStorage.getItem("flowxtra-token")) {
          response = await apiGet<any[]>(
            API_ENDPOINTS.GetAllPolicies,
            { enabled: "true" }, // Filter for enabled policies only
            subdomain
          );
        } else {
          throw publicError;
        }
      }

      if (response.success && response.data) {
        // Filter for published and enabled policies (if using authenticated endpoint)
        const policies = Array.isArray(response.data) ? response.data : [];
        const publishedPolicies = policies.filter(
          (p: any) => (p.is_published !== false && p.is_enabled !== false) || p.is_published === true
        );
        setPublishedPolicies(publishedPolicies);
      } else {
        setPublishedPolicies([]);
      }
    } catch (error) {
      console.error("Error loading policies:", error);
      setPublishedPolicies([]);
    }
  };

  // Get policy by type
  const getPolicyByType = (type: "privacy_policy" | "terms_of_service"): any => {
    if (!publishedPolicies || publishedPolicies.length === 0) {
      return null;
    }
    return publishedPolicies.find((p) => p.policy_type === type) || null;
  };

  // Get policy URL for a specific type
  const getPolicyUrlByType = (type: "terms" | "privacy"): string => {
    const policyType = type === "terms" ? "terms_of_service" : "privacy_policy";
    const policy = getPolicyByType(policyType);
    if (policy && policy.slug) {
      return `/policy/${policy.slug}`;
    }
    return "#";
  };

  const loadFilterData = async () => {
    if (!subdomain) return;
    try {
      const response = await apiGet<FilterData>(
        API_ENDPOINTS.CreateProfilFilter,
        undefined,
        subdomain
      );
      setFilterData(response.data);
    } catch (error) {
      console.error("Error loading filter data:", error);
    }
  };

  const loadProfile = async (page: number) => {
    if (!subdomain) return;
    try {
      setIsLoading(true);
      const params: Record<string, any> = {
        page: page.toString(),
        max: "10",
      };

      if (filterObj.catrgory_id) params["key-catrgory"] = filterObj.catrgory_id;
      if (filterObj.employment_type_id)
        params["key-employment-type"] = filterObj.employment_type_id;
      if (filterObj.location_id) params["key-location"] = filterObj.location_id;
      if (filterObj.department_id) params["key-department"] = filterObj.department_id;

      const response = await apiGet<ProfileData>(
        API_ENDPOINTS.Show,
        params,
        subdomain
      );

      setProfile(response.data.profile);
      setJobs(response.data.jobs.data);
      setCurrentPage(response.data.jobs.current_page);
      setTotalPages(response.data.jobs.last_page);

      if (response.data.profile?.company_image_gallery) {
        setCompanyImageGallery(response.data.profile.company_image_gallery);
      } else {
        setCompanyImageGallery([]);
      }

          if (response.data.profile?.company_show_social) {
            setCompanyShowSocial(response.data.profile.company_show_social);
          } else {
            setCompanyShowSocial(null);
          }

          // Load Brand Kit Settings
          const profileData = response.data.profile as any;
          console.log("🔍 [Brand Kit] loadProfile - Footer Settings:", profileData.footer_settings);
          console.log("🔍 [Brand Kit] loadProfile - Social Links:", profileData.social_links);
          console.log("🔍 [Brand Kit] loadProfile - Social Styling:", profileData.social_styling);
          
          if (profileData.typography_settings) {
            setTypographySettings(profileData.typography_settings);
          }
          if (profileData.logo_settings || profileData.logo_setting) {
            setLogoSettings(profileData.logo_settings || profileData.logo_setting);
          }
          if (profileData.footer_settings) {
            setFooterSettings(profileData.footer_settings);
          }
          if (profileData.social_links && Array.isArray(profileData.social_links)) {
            setSocialLinks(profileData.social_links);
          }
          if (profileData.social_styling) {
            setSocialStyling(profileData.social_styling);
          }

          setIsLoading(false);
    } catch (error) {
      console.error("Error loading profile:", error);
      setIsLoading(false);
    }
  };

  const goToPage = (page: number) => {
    if (page >= 1 && page <= totalPages) {
      setCurrentPage(page);
      loadProfile(page);
    }
  };

  const handleFilterChange = (key: keyof FilterObj, value: string) => {
    setFilterObj((prev) => ({
      ...prev,
      [key]: value === "" ? null : value,
    }));
  };

  if (isLoading || !profile) {
    return (
      <div className="flex min-h-screen items-center justify-center">
        <div className="text-center">
          <div className="loading-spinner"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    );
  }

  // Get current URL for canonical
  const canonicalUrl =
    typeof window !== "undefined"
      ? window.location.href.split("?")[0] // Remove query params
      : "";

  // Get policy data for footer links (before return to avoid IIFE in JSX)
  const termsPolicy = getPolicyByType("terms_of_service");
  const privacyPolicy = getPolicyByType("privacy_policy");
  const showTermsLink = (shouldShowFooterLink(footerSettings, "showTerms") || !footerSettings) && termsPolicy && termsPolicy.slug;
  const showPrivacyLink = (shouldShowFooterLink(footerSettings, "showPrivacy") || !footerSettings) && privacyPolicy && privacyPolicy.slug;

  return (
    <>
      <MetaTags profile={profile} canonicalUrl={canonicalUrl} logoSettings={logoSettings} />
      <div className="m-page">
        {/* Company Header */}
      <section>
        <div className="flex flex-wrap justify-start logo-wrapper">
          {!isTextLogo(logoSettings, "header") &&
            getLogoUrl(logoSettings, profile.logo, "header") &&
            companyShowSocial?.logo !== false && (
              <Image
                src={getLogoUrl(logoSettings, profile.logo, "header") || "/assets/images/Logo-loading.svg"}
                alt={`${profile.name} company logo`}
                width={200}
                height={200}
                className="img-comp"
                priority
                unoptimized
              />
            )}
          {isTextLogo(logoSettings, "header") &&
            getTextLogoContent(logoSettings, "header") &&
            companyShowSocial?.logo !== false && (
              <a
                href={getTextLogoContent(logoSettings, "header")?.linkUrl || "#"}
                target={getTextLogoContent(logoSettings, "header")?.linkTarget}
                className="text-logo-header"
                style={{
                  fontFamily: getTextLogoContent(logoSettings, "header")?.fontFamily,
                  fontSize: `${getTextLogoContent(logoSettings, "header")?.fontSize}px`,
                  color: getTextLogoContent(logoSettings, "header")?.color,
                }}
              >
                {getTextLogoContent(logoSettings, "header")?.text}
              </a>
            )}
        </div>
        {companyShowSocial?.show_name_company && (
          <h1 className="fs-30 mt-2 fw-semibold name-header">{profile.name}</h1>
        )}
      </section>

      {/* Company Info Header */}
      <div className="company-info-header flex flex-wrap justify-between mt-2 mb-2">
        <div className="flex gap-3 flex-wrap company-details">
          {companyShowSocial?.show_num_emloyees && (
            <div className="fs-16 company-info-badge">
              <i className="bi bi-people-fill color-p me-1"></i>
              <span>
                {profile.number_employees} Employees
              </span>
            </div>
          )}
          {profile.company_type && companyShowSocial?.show_category_company && (
            <div className="fs-16 company-info-badge">
              <i className="bi bi-grid color-p me-1"></i>
              <span>{profile.company_type.name}</span>
            </div>
          )}
        </div>
        {profile.show_website && (
          <div className="website-btn-wrapper">
            <a
              href={profile.website || "#"}
              target="_blank"
              rel="noopener noreferrer"
              className="btn-theme-outline cu-pointer"
            >
              Go to Website
            </a>
          </div>
        )}
      </div>

      {/* Company Description */}
      {companyShowSocial?.show_about_company && (
        <>
          {profile.description && (
            <div
              className="mt-2 gray-6 company-description"
              dangerouslySetInnerHTML={{ __html: profile.description }}
            />
          )}
          {profile.mission && (
            <div
              className="mt-2 gray-6 company-description"
              dangerouslySetInnerHTML={{ __html: profile.mission }}
            />
          )}
        </>
      )}

      {/* Navigation */}
      {(companyShowSocial?.category_filter ||
        companyShowSocial?.employment_type_filter ||
        companyShowSocial?.department_filter ||
        companyShowSocial?.location_filter ||
        companyShowSocial?.show_media ||
        companyShowSocial?.show_values ||
        companyShowSocial?.show_benfits) && (
        <nav className="navbar navbar-light bg-light my-5 sticky-top">
          <ul className="nav nav-pills">
            {(companyShowSocial?.category_filter ||
              companyShowSocial?.employment_type_filter ||
              companyShowSocial?.department_filter ||
              companyShowSocial?.location_filter) && (
              <li className="nav-item">
                <a className="nav-link color-p fs-18" href="#positions">
                  Open Positions
                </a>
              </li>
            )}
            {companyShowSocial?.show_benfits && (
              <li className="nav-item">
                <a className="nav-link color-p fs-18" href="#Benefits">
                  Benefits
                </a>
              </li>
            )}
            {companyShowSocial?.show_values && (
              <li className="nav-item">
                <a className="nav-link color-p fs-18" href="#Values">
                  Values
                </a>
              </li>
            )}
            {companyShowSocial?.show_media && (
              <li className="nav-item">
                <a className="nav-link color-p fs-18" href="#Media">
                  Media
                </a>
              </li>
            )}
          </ul>
        </nav>
      )}

      {/* Filters and Jobs Section */}
      {(companyShowSocial?.category_filter ||
        companyShowSocial?.employment_type_filter ||
        companyShowSocial?.department_filter ||
        companyShowSocial?.location_filter) && (
        <div className="row" id="positions">
          <div className="col-12">
            <h2 className="fs-20 fw-bold">Open Positions</h2>
          </div>

          {/* Employment Type Filter */}
          {companyShowSocial?.employment_type_filter && filterData && (
            <div className="col-lg-6 col-12 mt-3">
              <label htmlFor="employment-type" className="form-label fw-bold">
                Employment Type
              </label>
              <select
                className="form-control height pad-7"
                id="employment-type"
                value={filterObj.employment_type_id || ""}
                onChange={(e) => handleFilterChange("employment_type_id", e.target.value)}
              >
                <option value="">All</option>
                {filterData.emplement_types.map((item) => (
                  <option key={item.id} value={item.id}>
                    {item.name}
                  </option>
                ))}
              </select>
            </div>
          )}

          {/* Department Filter */}
          {companyShowSocial?.department_filter && filterData && (
            <div className="col-lg-6 col-12 mt-3">
              <label htmlFor="department" className="form-label fw-bold">
                Department
              </label>
              <select
                className="form-control height pad-7"
                id="department"
                value={filterObj.department_id || ""}
                onChange={(e) => handleFilterChange("department_id", e.target.value)}
              >
                <option value="">All</option>
                {filterData.company_show_department.map((item) => (
                  <option key={item.id} value={item.id}>
                    {item.name}
                  </option>
                ))}
              </select>
            </div>
          )}

          {/* Location Filter */}
          {companyShowSocial?.location_filter && filterData && (
            <div className="col-lg-12 col-12 mt-3">
              <label htmlFor="location" className="form-label fw-bold">
                Location
              </label>
              <select
                className="form-control height pad-7"
                id="location"
                value={filterObj.location_id || ""}
                onChange={(e) => handleFilterChange("location_id", e.target.value)}
              >
                <option value="">All</option>
                {filterData.locations.map((item) => (
                  <option key={item.id} value={item.id}>
                    {item.name}
                  </option>
                ))}
              </select>
            </div>
          )}
        </div>
      )}

      {/* Jobs List */}
      <div className="row">
        {jobs.map((item) => (
          <div key={item.id} className="col-12">
            <div className="card-list my-2 job-card">
              <div className="row">
                <div className="col-lg-10 col-sm-12">
                  {/* Job Title with Logo */}
                  <div className="flex items-center mb-2">
                    {!isTextLogo(logoSettings, "header") &&
                      getLogoUrl(logoSettings, profile.logo, "header") &&
                      companyShowSocial?.logo && (
                        <div className="me-2">
                          <Image
                            src={getLogoUrl(logoSettings, profile.logo, "header") || "/assets/images/Logo-loading.svg"}
                            alt={`${profile.name} company logo`}
                            width={50}
                            height={50}
                            className="rounded-full img-logo"
                            unoptimized
                          />
                        </div>
                      )}
                    {isTextLogo(logoSettings, "header") &&
                      getTextLogoContent(logoSettings, "header") &&
                      companyShowSocial?.logo && (
                        <div className="me-2">
                          <a
                            href={getTextLogoContent(logoSettings, "header")?.linkUrl || "#"}
                            target={getTextLogoContent(logoSettings, "header")?.linkTarget}
                            className="text-logo-small"
                            style={{
                              fontFamily: getTextLogoContent(logoSettings, "header")?.fontFamily,
                              fontSize: `${(getTextLogoContent(logoSettings, "header")?.fontSize || 24) * 0.6}px`,
                              color: getTextLogoContent(logoSettings, "header")?.color,
                            }}
                          >
                            {getTextLogoContent(logoSettings, "header")?.text}
                          </a>
                        </div>
                      )}
                    <div className="flex-grow-1">
                      <Link
                        href={`/job/${sanitizeTitle(item.title)}/${item.hash_id}`}
                        className="text-decoration-none"
                      >
                        <h3 className="fs-18 fw-bold color-p cu_pointer mb-0">
                          {item.title}
                        </h3>
                      </Link>
                    </div>
                  </div>

                  {/* Job Details - First Row */}
                  <div className="flex flex-wrap job-info-row">
                    {companyShowSocial?.show_job_workplace && (
                      <div className="job-info-item mt-10 me-2">
                        <i className="color-p bi bi-pin-map-fill me-1"></i>
                        <span>{item.workplace}</span>
                      </div>
                    )}
                    {companyShowSocial?.show_category_company &&
                      item.company_job_category && (
                        <div className="job-info-item mt-10 me-2">
                          <i className="color-p bi bi-shop-window me-1"></i>
                          <span>{item.company_job_category.name}</span>
                        </div>
                      )}
                    {companyShowSocial?.show_date_job && (
                      <div className="job-info-item mt-10 me-2">
                        <i className="color-p bi bi-calendar2-event-fill me-1"></i>
                        <span>
                          {new Date(item.created_at).toLocaleDateString("en-US", {
                            year: "numeric",
                            month: "short",
                            day: "numeric",
                          })}
                        </span>
                      </div>
                    )}
                  </div>

                  {/* Job Details - Second Row */}
                  <div className="flex flex-wrap job-info-row">
                    {(companyShowSocial?.employment_type === true ||
                      companyShowSocial?.employment_type === null) && (
                      <div className="job-info-item mt-10 me-2">
                        <i className="color-p bi bi-briefcase-fill me-1"></i>
                        <span>
                          {item.job_type_job.map((type, ind) => (
                            <span key={type.id}>
                              {ind !== 0 ? " / " : ""}
                              {type.job_type.name}
                            </span>
                          ))}
                        </span>
                      </div>
                    )}
                    {companyShowSocial?.salary && (
                      <div className="job-info-item mt-10 me-2">
                        <i className="color-p bi bi-cash-stack me-1"></i>
                        <span>
                          {formatSalary(
                            item.salary,
                            item.min_salary,
                            item.max_salary,
                            item.currency,
                            item.rate_salary,
                            item.salry_pay_by
                          )}
                        </span>
                      </div>
                    )}
                    {(companyShowSocial?.location === true ||
                      companyShowSocial?.location === null) && (
                      <div className="job-info-item mt-10 me-2">
                        <i className="color-p bi bi-geo-alt me-1"></i>
                        <span>
                          {item.company_office.country.name} /{" "}
                          {item.company_office.state.name}
                        </span>
                      </div>
                    )}
                  </div>
                </div>

                {/* Apply Button */}
                <div className="col-lg-2 col-sm-12 flex items-center apply-btn-container">
                  <Link
                    href={`/job/${sanitizeTitle(item.title)}/${item.hash_id}`}
                    className="btn-theme py-2 px-3"
                  >
                    Apply <i className="bi bi-chevron-right"></i>
                  </Link>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Pagination */}
      <div className="col-lg-12 col-sm-12 mt-1 flex justify-center">
        <div className="page-group">
          <button
            className="page-item"
            onClick={() => goToPage(currentPage - 1)}
            disabled={currentPage === 1}
            type="button"
          >
            <i className="bi bi-chevron-left"></i>
          </button>
          <span>
            Page {currentPage} of {totalPages}
          </span>
          <button
            className="page-item"
            onClick={() => goToPage(currentPage + 1)}
            disabled={currentPage === totalPages}
            type="button"
          >
            <i className="bi bi-chevron-right"></i>
          </button>
        </div>
      </div>

      {/* Benefits Section */}
      {companyShowSocial?.show_benfits && profile.benefits_company?.length > 0 && (
        <div className="container overflow-hidden" id="Benefits">
          <div className="col-12">
            <h2 className="fs-20 fw-bold">Benefits</h2>
          </div>
          <div className="row gx-5 mt-2">
            {profile.benefits_company.map((item) => (
              <div key={item.id} className="col-lg-6 col-sm-12">
                <div className="p-2 border bg-light my-2 flex flex-row color-p rounded-3">
                  <i className="bi bi-check fs-22"></i>
                  <span>{item.name}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Values Section */}
      {companyShowSocial?.show_values && profile.company_value?.length > 0 && (
        <div className="container overflow-hidden mt-4" id="Values">
          <div className="col-12">
            <h2 className="fs-20 fw-bold">Values</h2>
          </div>
          <div className="row gx-5 mt-2">
            {profile.company_value.map((item) => (
              <div key={item.id} className="col-lg-6 col-sm-12">
                <div className="p-2 border bg-light my-2 flex flex-row color-p rounded-3">
                  <i className="bi bi-check fs-22"></i>
                  <span>{item.name}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Media Section */}
      {companyShowSocial?.show_media && companyImageGallery.length > 0 && (
        <div className="row mt-4" id="Media">
          <div className="col-12">
            <h2 className="fs-20 fw-bold">Media</h2>
          </div>
          {companyImageGallery.map((file, ind) => (
            <div key={ind} className="col-lg-6 col-sm-12">
              <div className="text-center text-lg m-10">
                <Image
                  src={file.imge}
                  alt={`Gallery image ${ind + 1} from ${profile.name}`}
                  width={600}
                  height={400}
                  className="image-container"
                  unoptimized
                />
              </div>
            </div>
          ))}
        </div>
      )}

      {/* Footer Custom Text */}
      {footerSettings && (footerSettings.customText || footerSettings.custom_text) && (
        <div className="footer-custom-text mb-3">
          <div
            className="custom-text-content"
            dangerouslySetInnerHTML={{
              __html: footerSettings.customText || footerSettings.custom_text || "",
            }}
          />
        </div>
      )}

      {/* Footer Links */}
      {(() => {
        console.log("🔗 [Footer] Rendering Footer Links section");
        console.log("🔗 [Footer] footerSettings:", footerSettings);
        console.log("🔗 [Footer] socialLinks:", socialLinks);
        console.log("🔗 [Footer] socialLinks length:", socialLinks?.length);
        console.log("🔗 [Footer] socialStyling:", socialStyling);
        return null;
      })()}
      <div className="flex flex-col items-center mt-4 mb-4">
        <div className="flex justify-center flex-wrap items-center gap-2">
          {/* Always show footer links by default if footerSettings is null, or show based on settings */}
          {showTermsLink && (
            <a
              href={`/policy/${termsPolicy.slug}`}
              onClick={(e) => {
                e.preventDefault();
                e.stopPropagation();
                try {
                  router.push(`/policy/${termsPolicy.slug}`);
                } catch (error) {
                  console.error("Navigation error:", error);
                  // Fallback to window.location if router.push fails
                  window.location.href = `/policy/${termsPolicy.slug}`;
                }
              }}
              className="fs-14 color-p text-decoration-none d-flex align-items-center gap-1 cursor-pointer"
              style={{ cursor: 'pointer' }}
            >
              Terms & Conditions
            </a>
          )}
          {showPrivacyLink && (
            <a
              href={`/policy/${privacyPolicy.slug}`}
              onClick={(e) => {
                e.preventDefault();
                e.stopPropagation();
                try {
                  router.push(`/policy/${privacyPolicy.slug}`);
                } catch (error) {
                  console.error("Navigation error:", error);
                  // Fallback to window.location if router.push fails
                  window.location.href = `/policy/${privacyPolicy.slug}`;
                }
              }}
              className="fs-14 color-p text-decoration-none d-flex align-items-center gap-1 cursor-pointer"
              style={{ cursor: 'pointer' }}
            >
              Privacy Policy
            </a>
          )}
          {(shouldShowFooterLink(footerSettings, "showReport") || !footerSettings) && (
              <button
                type="button"
                className="fs-14 color-p text-decoration-none bg-transparent border-0 p-0 cursor-pointer d-flex align-items-center gap-1"
                onClick={() => {
                  openReportIssue();
                }}
              >
                <i className="bi bi-flag-fill" aria-hidden="true"></i>
                <span>Report Issue</span>
              </button>
          )}
          {(shouldShowFooterLink(footerSettings, "showAccessibility") || !footerSettings) && (
              <a
                className="fs-14 color-p text-decoration-none d-flex align-items-center gap-1"
                href="#"
                onClick={(e) => {
                  e.preventDefault();
                  // Accessibility widget functionality - can be implemented later
                  alert("Accessibility options coming soon");
                }}
              >
                <i className="bi bi-universal-access-circle" aria-hidden="true"></i>
                <span>Accessibility</span>
              </a>
          )}
          {(shouldShowFooterLink(footerSettings, "showSitemap") || !footerSettings) && (
              <a
                className="fs-14 color-p text-decoration-none d-flex align-items-center gap-1"
                target="_blank"
                rel="noopener noreferrer"
                href={`${getUrlSitemap(subdomain)}sitemap.xml`}
                title="Sitemap"
              >
                <i className="bi bi-diagram-3-fill" aria-hidden="true"></i>
                <span>Sitemap</span>
              </a>
          )}
        </div>

        {/* Social Media Links */}
          {(() => {
            console.log("🔗 [Social] Checking social links - socialLinks:", socialLinks);
            console.log("🔗 [Social] socialLinks length:", socialLinks?.length);
            console.log("🔗 [Social] socialStyling:", socialStyling);
            return null;
          })()}
          {socialLinks && socialLinks.length > 0 && (
            <div
              className="social-links-container flex justify-center items-center flex-wrap"
              style={{ gap: `${socialStyling?.spacing || 12}px` }}
            >
              {socialLinks.map((link, index) => {
                const icon = getSocialLinkIcon(link);
                const iconStyles = getSocialLinkStyles(link, socialStyling);
                const iconColorStyle = getIconColorStyle(link, socialStyling);
                const hoverClass = getSocialLinkClass(socialStyling);
                console.log(`🔗 [Social] Rendering link ${index}:`, { icon, iconStyles, iconColorStyle, hoverClass, link });

                return (
                  <a
                    key={index}
                    href={link.url || link.link_url || "#"}
                    target="_blank"
                    rel="noopener noreferrer"
                    title={getSocialLinkName(link)}
                    style={iconStyles}
                    className={`social-link-item flex items-center justify-center text-decoration-none ${hoverClass}`}
                  >
                    {isFontAwesome(icon) ? (
                      <i className={icon} style={iconColorStyle}></i>
                    ) : (
                      <span style={iconColorStyle}>{icon}</span>
                    )}
                  </a>
                );
            })}
          </div>
        )}
      </div>

      {/* Flowxtra Branding */}
      {shouldShowFooterLink(footerSettings, "showBranding") && (
        <div className="flex justify-center items-center flex-wrap m-5">
          <span className="fs-16 color-p text-center">We Hire With</span>
          <a
            className="fs-16 color-p ms-1 text-decoration-none fw-bold text-center"
            target="_blank"
            rel="noopener noreferrer"
            href="https://flowxtra.com"
          >
            Flowxtra
          </a>
        </div>
      )}

      {/* Report Issue Modal */}
      <ReportIssueModal
        isOpen={isReportModalOpen}
        onClose={handleReportModalClose}
        data={{
          companyName: profile?.name,
          companyId: profile?.id,
        } as ReportIssueData}
      />
      </div>
    </>
  );
}

