// Dynamic metadata generation for profile page
import type { Metadata } from "next";
import type { Company } from "@/lib/types";

export function generateProfileMetadata(
  profile: Company | null,
  canonicalUrl?: string
): Metadata {
  const companyName = profile?.name || "Company";
  const description = profile?.description
    ? profile.description.replace(/<[^>]+>/g, "").substring(0, 160)
    : `Explore career opportunities at ${companyName}. Find your perfect job match.`;

  const keywords = [
    companyName,
    "company profile",
    "jobs",
    "careers",
    "recruitment",
    "Flowxtra",
  ];

  if (profile?.company_type?.name) {
    keywords.push(profile.company_type.name);
  }

  if (profile?.number_employees) {
    keywords.push(`${profile.number_employees} employees`);
  }

  const metadata: Metadata = {
    title: `${companyName} - Career`,
    description: description,
    keywords: keywords,
    openGraph: {
      title: `${companyName} - Career`,
      description: description,
      type: "website",
      images: profile?.logo
        ? [
            {
              url: profile.logo,
              width: 1200,
              height: 630,
              alt: `${companyName} logo`,
            },
          ]
        : [],
    },
    twitter: {
      card: "summary_large_image",
      title: `${companyName} - Career`,
      description: description,
      images: profile?.logo ? [profile.logo] : [],
    },
  };

  // Add canonical URL if provided
  if (canonicalUrl) {
    metadata.alternates = {
      canonical: canonicalUrl,
    };
  }

  return metadata;
}
