"use client";

import { useEffect } from "react";
import type { Company } from "@/lib/types";

interface MetaTagsProps {
  profile: Company | null;
  canonicalUrl: string;
}

export default function MetaTags({ profile, canonicalUrl }: MetaTagsProps) {
  useEffect(() => {
    if (!profile) return;

    const companyName = profile.name || "Company";
    const description = profile.description
      ? profile.description.replace(/<[^>]+>/g, "").substring(0, 160)
      : `Explore career opportunities at ${companyName}. Find your perfect job match.`;

    // Update title
    document.title = `${companyName} - Career`;

    // Update or create meta tags
    const updateMetaTag = (name: string, content: string, property = false) => {
      const selector = property ? `property="${name}"` : `name="${name}"`;
      let meta = document.querySelector(`meta[${selector}]`);
      
      if (!meta) {
        meta = document.createElement("meta");
        if (property) {
          meta.setAttribute("property", name);
        } else {
          meta.setAttribute("name", name);
        }
        document.head.appendChild(meta);
      }
      meta.setAttribute("content", content);
    };

    // Standard meta tags
    updateMetaTag("description", description);
    updateMetaTag("keywords", [
      companyName,
      "company profile",
      "jobs",
      "careers",
      "recruitment",
      "Flowxtra",
      profile.company_type?.name,
      profile.number_employees ? `${profile.number_employees} employees` : "",
    ]
      .filter(Boolean)
      .join(", "));

    // Open Graph meta tags
    updateMetaTag("og:title", `${companyName} - Career`, true);
    updateMetaTag("og:description", description, true);
    updateMetaTag("og:type", "website", true);
    updateMetaTag("og:url", canonicalUrl, true);
    if (profile.logo) {
      updateMetaTag("og:image", profile.logo, true);
    }

    // Twitter Card meta tags
    updateMetaTag("twitter:card", "summary_large_image", true);
    updateMetaTag("twitter:title", `${companyName} - Career`, true);
    updateMetaTag("twitter:description", description, true);
    if (profile.logo) {
      updateMetaTag("twitter:image", profile.logo, true);
    }

    // Canonical URL
    let canonicalLink = document.querySelector('link[rel="canonical"]');
    if (!canonicalLink) {
      canonicalLink = document.createElement("link");
      canonicalLink.setAttribute("rel", "canonical");
      document.head.appendChild(canonicalLink);
    }
    canonicalLink.setAttribute("href", canonicalUrl);

    // JSON-LD Structured Data
    let jsonLdScript = document.getElementById("company-json-ld") as HTMLScriptElement | null;
    if (jsonLdScript) {
      jsonLdScript.remove();
    }
    jsonLdScript = document.createElement("script");
    jsonLdScript.id = "company-json-ld";
    jsonLdScript.type = "application/ld+json";

    const jsonLd = {
      "@context": "https://schema.org",
      "@type": "Organization",
      name: companyName,
      description: description,
      url: canonicalUrl,
      ...(profile.logo && {
        logo: {
          "@type": "ImageObject",
          url: profile.logo,
        },
      }),
      ...(profile.website && {
        sameAs: [profile.website],
      }),
      ...(profile.number_employees && {
        numberOfEmployees: {
          "@type": "QuantitativeValue",
          value: profile.number_employees,
        },
      }),
      ...(profile.company_type && {
        industry: profile.company_type.name,
      }),
      ...(profile.benefits_company && profile.benefits_company.length > 0 && {
        employeeBenefits: profile.benefits_company.map((benefit) => benefit.name),
      }),
    };

    jsonLdScript.textContent = JSON.stringify(jsonLd);
    document.head.appendChild(jsonLdScript);
  }, [profile, canonicalUrl]);

  return null;
}

