import { NextRequest, NextResponse } from "next/server";
import xmlbuilder from "xmlbuilder";
import { API_BASE_URL, API_ENDPOINTS } from "@/lib/api";
import { getOriginalHost, getSubdomainFromHost } from "@/lib/utils-server";

function safeAddElement(parent: any, elementName: string, value: any): void {
  if (value !== undefined && value !== null && value !== "") {
    try {
      const sanitizedValue = String(value).replace(/[<>&]/g, (match) => {
        const escapeMap: { [key: string]: string } = {
          "<": "&lt;",
          ">": "&gt;",
          "&": "&amp;",
        };
        return escapeMap[match];
      });
      parent.ele(elementName, sanitizedValue);
    } catch (error) {
      console.warn(`Failed to add element ${elementName}:`, error);
    }
  }
}

export async function GET(request: NextRequest) {
  try {
    const headers = request.headers;
    const host = getOriginalHost(headers);
    const subdomain = getSubdomainFromHost(host) || "default";

    const apiUrl = `${API_BASE_URL}${API_ENDPOINTS.JobCompanyFeed}?subdomain=${subdomain}`;

    const response = await fetch(apiUrl, {
      headers: {
        "User-Agent": "Flowxtra-RSS-Generator/1.0",
        Accept: "application/json",
        Authorization: headers.get("authorization") || "",
      },
      cache: "no-store",
    });

    const jobs = response.ok ? (await response.json())?.data || [] : [];

    // Create RSS feed structure
    const rss = xmlbuilder.create("rss", {
      version: "1.0",
      encoding: "UTF-8",
    });
    rss.att("version", "2.0");
    rss.att("xmlns:atom", "http://www.w3.org/2005/Atom");

    // Add channel information
    const channel = rss.ele("channel");
    channel.ele("title", "Flowxtra Jobs Feed");
    channel.ele("description", "Latest job opportunities from Flowxtra");
    channel.ele("language", "en-US");
    channel.ele("lastBuildDate", new Date().toISOString());
    channel.ele("pubDate", new Date().toISOString());
    channel.ele("ttl", "60");

    // Add atom:link for RSS autodiscovery
    const atomLink = channel.ele("atom:link");
    const proto =
      headers.get("x-forwarded-proto")?.split(",")[0].trim() || "https";
    const baseUrl = `${proto.replace(/:$/, "")}://${host}`;
    atomLink.att("href", `${baseUrl}/feed.xml`);
    atomLink.att("rel", "self");
    atomLink.att("type", "application/rss+xml");

    // Add job items to the feed
    jobs.forEach((job: any) => {
      try {
        const item = channel.ele("item");

        safeAddElement(item, "title", job.title);
        safeAddElement(item, "link", job.link);
        safeAddElement(item, "guid", job.link);
        safeAddElement(
          item,
          "pubDate",
          new Date(job.created_at || Date.now()).toISOString()
        );

        // Create description with job details using CDATA
        let description = `<h3>${job.title || "Job Title"}</h3>`;
        if (job.company_name) {
          description += `<p><strong>Company:</strong> ${job.company_name}</p>`;
        }
        if (job.country || job.state) {
          const location = [job.state, job.country].filter(Boolean).join(", ");
          if (location) {
            description += `<p><strong>Location:</strong> ${location}</p>`;
          }
        }
        if (job.address) {
          description += `<p><strong>Address:</strong> ${job.address}</p>`;
        }
        if (job.summary) {
          description += `<p><strong>Summary:</strong> ${job.summary}</p>`;
        }
        if (job.description) {
          description += `<p><strong>Description:</strong> ${job.description}</p>`;
        }

        item.ele("description").dat(description);

        // Add structured job data fields
        safeAddElement(item, "company", job.company_name);

        if (job.country || job.state) {
          const location = [job.state, job.country].filter(Boolean).join(", ");
          safeAddElement(item, "location", location);
        }

        safeAddElement(item, "address", job.address);
        safeAddElement(item, "employmentType", job.jobType);
        safeAddElement(item, "jobId", job.id);
        safeAddElement(item, "industry", job.jobCategory);
        safeAddElement(item, "jobFunction", job.companyJobCategory);

        // Create baseSalary from min and max salary
        if (job.min_salary !== undefined && job.max_salary !== undefined) {
          const baseSalary = `${job.min_salary} - ${job.max_salary}`;
          if (job.currency) {
            safeAddElement(item, "baseSalary", `${baseSalary} ${job.currency}`);
          } else {
            safeAddElement(item, "baseSalary", baseSalary);
          }
        } else if (job.salary !== undefined) {
          if (job.currency) {
            safeAddElement(item, "baseSalary", `${job.salary} ${job.currency}`);
          } else {
            safeAddElement(item, "baseSalary", job.salary);
          }
        }

        safeAddElement(item, "currency", job.currency);
        safeAddElement(item, "country", job.country);
        safeAddElement(item, "state", job.state);
      } catch (error) {
        console.error("Error processing job item:", error);
      }
    });

    return new NextResponse(rss.end({ pretty: true }), {
      status: 200,
      headers: {
        "Content-Type": "application/rss+xml",
      },
    });
  } catch (error) {
    console.error("Error generating RSS feed:", error);

    // Return a basic RSS feed structure even if API fails
    const rss = xmlbuilder.create("rss", {
      version: "1.0",
      encoding: "UTF-8",
    });
    rss.att("version", "2.0");
    rss.att("xmlns:atom", "http://www.w3.org/2005/Atom");

    const channel = rss.ele("channel");
    channel.ele("title", "Flowxtra Jobs Feed");
    channel.ele("description", "Latest job opportunities from Flowxtra");
    channel.ele("link", process.env.NEXT_PUBLIC_API_BASE_URL || "");
    channel.ele("language", "en-US");
    channel.ele("lastBuildDate", new Date().toISOString());
    channel.ele("pubDate", new Date().toISOString());
    channel.ele("ttl", "60");

    const atomLink = channel.ele("atom:link");
    atomLink.att(
      "href",
      `${process.env.NEXT_PUBLIC_API_BASE_URL || ""}/feed.xml`
    );
    atomLink.att("rel", "self");
    atomLink.att("type", "application/rss+xml");

    return new NextResponse(rss.end({ pretty: true }), {
      status: 200,
      headers: {
        "Content-Type": "application/rss+xml",
      },
    });
  }
}
