"use client";

import { useState, useEffect } from "react";
import styles from "./ReportIssueModal.module.css";

export interface ReportIssueData {
  companyName?: string;
  jobTitle?: string;
  jobId?: string | number;
  workplace?: string;
  companyId?: string | number;
}

interface IssueCategory {
  value: number;
  key: string;
  label: string;
  hasSubCategories: boolean;
}

interface IssueSubCategory {
  value: number;
  key: string;
  label: string;
  parentCategory: number;
}

interface ReportIssueModalProps {
  isOpen: boolean;
  onClose: (result?: { success: boolean; data?: any; error?: any }) => void;
  data?: ReportIssueData;
}

export default function ReportIssueModal({
  isOpen,
  onClose,
  data,
}: ReportIssueModalProps) {
  // Main categories with labels
  const mainCategories: IssueCategory[] = [
    { value: 1, key: "REPORT_CATEGORY_1", label: "Fraudulent Activity", hasSubCategories: true },
    { value: 2, key: "REPORT_CATEGORY_2", label: "Inaccurate Information", hasSubCategories: true },
    { value: 3, key: "REPORT_CATEGORY_3", label: "Discriminatory Practices", hasSubCategories: true },
    { value: 4, key: "REPORT_CATEGORY_4", label: "Policy Violations", hasSubCategories: true },
    { value: 5, key: "REPORT_CATEGORY_5", label: "Technical Problems", hasSubCategories: true },
    { value: 6, key: "REPORT_CATEGORY_6", label: "Other Concerns", hasSubCategories: false },
  ];

  // Sub-categories for each main category
  const subCategories: IssueSubCategory[] = [
    // Fraudulent Activity (1)
    { value: 101, key: "REPORT_SUB_1_1", label: "Fake job posting", parentCategory: 1 },
    { value: 102, key: "REPORT_SUB_1_2", label: "Phishing or scam", parentCategory: 1 },
    { value: 103, key: "REPORT_SUB_1_3", label: "Identity theft", parentCategory: 1 },
    { value: 104, key: "REPORT_SUB_1_4", label: "Payment fraud", parentCategory: 1 },
    { value: 105, key: "REPORT_SUB_1_5", label: "Other fraudulent activity", parentCategory: 1 },

    // Inaccurate Information (2)
    { value: 201, key: "REPORT_SUB_2_1", label: "Incorrect job description", parentCategory: 2 },
    { value: 202, key: "REPORT_SUB_2_2", label: "Wrong salary information", parentCategory: 2 },
    { value: 203, key: "REPORT_SUB_2_3", label: "Misleading location", parentCategory: 2 },
    { value: 204, key: "REPORT_SUB_2_4", label: "False company information", parentCategory: 2 },
    { value: 205, key: "REPORT_SUB_2_5", label: "Other inaccurate information", parentCategory: 2 },

    // Discriminatory Practices (3)
    { value: 301, key: "REPORT_SUB_3_1", label: "Age discrimination", parentCategory: 3 },
    { value: 302, key: "REPORT_SUB_3_2", label: "Gender discrimination", parentCategory: 3 },
    { value: 303, key: "REPORT_SUB_3_3", label: "Race or ethnicity discrimination", parentCategory: 3 },
    { value: 304, key: "REPORT_SUB_3_4", label: "Religious discrimination", parentCategory: 3 },
    { value: 305, key: "REPORT_SUB_3_5", label: "Disability discrimination", parentCategory: 3 },
    { value: 306, key: "REPORT_SUB_3_6", label: "Sexual orientation discrimination", parentCategory: 3 },
    { value: 307, key: "REPORT_SUB_3_7", label: "Other discriminatory practices", parentCategory: 3 },

    // Policy Violations (4)
    { value: 401, key: "REPORT_SUB_4_1", label: "Spam or unsolicited messages", parentCategory: 4 },
    { value: 402, key: "REPORT_SUB_4_2", label: "Harassment", parentCategory: 4 },
    { value: 403, key: "REPORT_SUB_4_3", label: "Inappropriate content", parentCategory: 4 },
    { value: 404, key: "REPORT_SUB_4_4", label: "Terms of service violation", parentCategory: 4 },
    { value: 405, key: "REPORT_SUB_4_5", label: "Other policy violations", parentCategory: 4 },

    // Technical Problems (5)
    { value: 501, key: "REPORT_SUB_5_1", label: "Page not loading", parentCategory: 5 },
    { value: 502, key: "REPORT_SUB_5_2", label: "Broken links", parentCategory: 5 },
    { value: 503, key: "REPORT_SUB_5_3", label: "Application submission error", parentCategory: 5 },
    { value: 504, key: "REPORT_SUB_5_4", label: "Other technical problems", parentCategory: 5 },
  ];

  const [selectedMainCategory, setSelectedMainCategory] = useState<number | null>(null);
  const [selectedSubCategory, setSelectedSubCategory] = useState<number | null>(null);
  const [email, setEmail] = useState("");
  const [details, setDetails] = useState("");
  const [showSubCategories, setShowSubCategories] = useState(false);
  const [showDetailsForm, setShowDetailsForm] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Reset form when modal closes
  useEffect(() => {
    if (!isOpen) {
      setSelectedMainCategory(null);
      setSelectedSubCategory(null);
      setEmail("");
      setDetails("");
      setShowSubCategories(false);
      setShowDetailsForm(false);
      setIsSubmitting(false);
    }
  }, [isOpen]);

  /**
   * Select a main category
   */
  const selectMainCategory = (category: IssueCategory) => {
    setSelectedMainCategory(category.value);

    if (category.hasSubCategories) {
      setShowSubCategories(true);
    } else {
      // Category 6 (Other) goes directly to form
      setShowDetailsForm(true);
    }
  };

  /**
   * Select a sub-category
   */
  const selectSubCategory = (subCategory: number) => {
    setSelectedSubCategory(subCategory);
    setShowDetailsForm(true);
  };

  /**
   * Get filtered sub-categories for selected main category
   */
  const getFilteredSubCategories = (): IssueSubCategory[] => {
    if (!selectedMainCategory) return [];
    return subCategories.filter(
      (sub) => sub.parentCategory === selectedMainCategory
    );
  };

  /**
   * Go back to main categories
   */
  const goBackToMain = () => {
    setSelectedMainCategory(null);
    setSelectedSubCategory(null);
    setShowSubCategories(false);
    setShowDetailsForm(false);
  };

  /**
   * Go back to sub-categories
   */
  const goBackToSub = () => {
    setSelectedSubCategory(null);
    setShowDetailsForm(false);
    setShowSubCategories(true);
  };

  /**
   * Close the modal
   */
  const handleClose = () => {
    onClose();
  };

  /**
   * Check if form is valid
   */
  const isFormValid = (): boolean => {
    return !!selectedMainCategory && details.trim().length > 0;
  };

  /**
   * Get character count
   */
  const getCharacterCount = (): number => {
    return details.length;
  };

  /**
   * Get selected main category label
   */
  const getSelectedMainCategoryLabel = (): string => {
    const category = mainCategories.find(
      (c) => c.value === selectedMainCategory
    );
    return category ? category.label : "";
  };

  /**
   * Get selected sub-category label
   */
  const getSelectedSubCategoryLabel = (): string => {
    const subCategory = subCategories.find(
      (s) => s.value === selectedSubCategory
    );
    return subCategory ? subCategory.label : "";
  };

  /**
   * Submit the report
   */
  const handleSubmit = async () => {
    if (!isFormValid() || isSubmitting) {
      return;
    }

    setIsSubmitting(true);

    try {
      // Prepare context data
      const context: any = {};
      if (data?.companyName) {
        context.company_name = data.companyName;
      }
      if (data?.jobTitle) {
        context.job_title = data.jobTitle;
      }
      if (data?.jobId) {
        context.job_id = data.jobId;
      }
      if (data?.workplace) {
        context.workplace = data.workplace;
      }
      if (data?.companyId) {
        context.company_id = data.companyId;
      }
      if (typeof window !== "undefined") {
        context.url = window.location.href;
      }

      // Prepare API payload
      const payload = {
        main_category_id: selectedMainCategory,
        sub_category_id: selectedSubCategory || null,
        details: details.trim(),
        email: email.trim() || null,
        context: Object.keys(context).length > 0 ? context : null,
      };

      // Send to central API
      // Note: We need to use the central API endpoint, not the regular API
      const centralApiUrl = process.env.NEXT_PUBLIC_CENTRAL_API_URL || "https://app.flowxtra.com";
      const apiUrl = `${centralApiUrl}/api/central/reports`;
      
      // Get token from localStorage if available
      const headers: HeadersInit = {
        "Content-Type": "application/json",
      };
      
      if (typeof window !== "undefined") {
        const token = localStorage.getItem("flowxtra-token");
        if (token && token.trim() !== "") {
          headers["Authorization"] = `Bearer ${token}`;
        }
      }
      
      const response = await fetch(apiUrl, {
        method: "POST",
        headers,
        body: JSON.stringify(payload),
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || "Failed to submit report");
      }

      const result = await response.json();
      onClose({ success: true, data: result });
    } catch (error: any) {
      console.error("Error submitting report:", error);
      onClose({ success: false, error: error.message || "Failed to submit report" });
    } finally {
      setIsSubmitting(false);
    }
  };

  if (!isOpen) return null;

  return (
    <div className={styles.reportIssueModalOverlay} onClick={handleClose}>
      <div className={styles.reportIssueModal} onClick={(e) => e.stopPropagation()}>
        {/* Main Categories Selection */}
        {!showSubCategories && !showDetailsForm && (
          <>
            <div className={styles.modalHeader}>
              <h2 className={styles.modalTitle}>Report Issue</h2>
              <button
                type="button"
                className={styles.closeBtn}
                onClick={handleClose}
                aria-label="Close"
              >
                <i className="bi bi-x-lg"></i>
              </button>
            </div>

            <div className={styles.modalBody}>
              <p className={styles.confidentialNotice}>
                Your report will be kept confidential. We take all reports seriously and will investigate accordingly.
              </p>

              <div className={styles.issueTypes}>
                {mainCategories.map((category) => (
                  <button
                    key={category.value}
                    type="button"
                    className={styles.issueTypeBtn}
                    onClick={() => selectMainCategory(category)}
                  >
                    {category.label}
                  </button>
                ))}
              </div>
            </div>

            <div className={styles.modalFooter}>
              <button type="button" className={styles.btnCancel} onClick={handleClose}>
                Cancel
              </button>
            </div>
          </>
        )}

        {/* Sub-Categories Selection */}
        {showSubCategories && !showDetailsForm && (
          <>
            <div className={styles.modalHeader}>
              <h2 className={styles.modalTitle}>Report Issue</h2>
              <button
                type="button"
                className={styles.closeBtn}
                onClick={handleClose}
                aria-label="Close"
              >
                <i className="bi bi-x-lg"></i>
              </button>
            </div>

            <div className={styles.modalBody}>
              <p className={styles.confidentialNotice}>
                Your report will be kept confidential. We take all reports seriously and will investigate accordingly.
              </p>

              {/* Selected Main Category */}
              <div className={styles.selectedIssueType}>
                <span className={styles.selectedLabel}>
                  {getSelectedMainCategoryLabel()}
                </span>
                <i className={`bi bi-check-circle ${styles.checkIcon}`}></i>
              </div>

              <div className={styles.issueTypes}>
                {getFilteredSubCategories().map((subCategory) => (
                  <button
                    key={subCategory.value}
                    type="button"
                    className={styles.issueTypeBtn}
                    onClick={() => selectSubCategory(subCategory.value)}
                  >
                    {subCategory.label}
                  </button>
                ))}
              </div>
            </div>

            <div className={styles.modalFooter}>
              <button type="button" className={styles.btnBack} onClick={goBackToMain}>
                <i className="bi bi-arrow-left"></i>
                Back
              </button>
              <button type="button" className={styles.btnCancel} onClick={handleClose}>
                Cancel
              </button>
            </div>
          </>
        )}

        {/* Details Form */}
        {showDetailsForm && (
          <>
            <div className={styles.modalHeader}>
              <h2 className={styles.modalTitle}>Report Issue</h2>
              <button
                type="button"
                className={styles.closeBtn}
                onClick={handleClose}
                aria-label="Close"
              >
                <i className="bi bi-x-lg"></i>
              </button>
            </div>

            <div className={styles.modalBody}>
              <p className={styles.confidentialNotice}>
                Your report will be kept confidential. We take all reports seriously and will investigate accordingly.
              </p>

              {/* Selected Categories */}
              <div className={styles.selectedCategories}>
                <div className={styles.selectedIssueType}>
                  <span className={styles.selectedLabel}>
                    {getSelectedMainCategoryLabel()}
                  </span>
                </div>
                {selectedSubCategory && (
                  <div className={styles.selectedIssueType}>
                    <span className={styles.selectedLabel}>
                      {getSelectedSubCategoryLabel()}
                    </span>
                    <i className={`bi bi-check-circle ${styles.checkIcon}`}></i>
                  </div>
                )}
              </div>

              {/* Email Field (Optional) */}
              <div className={styles.formGroup}>
                <label htmlFor="email">
                  Email Address
                  <span className={styles.optionalLabel}>(Optional)</span>
                </label>
                <input
                  type="email"
                  id="email"
                  className={styles.formControl}
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  placeholder="your.email@example.com"
                />
              </div>

              {/* Details Field (Required) */}
              <div className={styles.formGroup}>
                <label htmlFor="details">
                  Details
                  <span className={styles.requiredLabel}>*</span>
                </label>
                <div className={styles.characterCount}>
                  {getCharacterCount()} / 1000
                </div>
                <textarea
                  id="details"
                  className={`${styles.formControl} ${styles.textarea}`}
                  rows={6}
                  maxLength={1000}
                  value={details}
                  onChange={(e) => setDetails(e.target.value)}
                  placeholder="Please provide as much detail as possible about the issue..."
                />
              </div>
            </div>

            <div className={styles.modalFooter}>
              <button
                type="button"
                className={styles.btnBack}
                onClick={selectedSubCategory ? goBackToSub : goBackToMain}
              >
                <i className="bi bi-arrow-left"></i>
                Back
              </button>
              <div className={styles.footerActions}>
                <button type="button" className={styles.btnCancel} onClick={handleClose}>
                  Cancel
                </button>
                <button
                  type="button"
                  className={styles.btnSubmit}
                  disabled={!isFormValid() || isSubmitting}
                  onClick={handleSubmit}
                >
                  {isSubmitting ? "Submitting..." : "Submit"}
                </button>
              </div>
            </div>
          </>
        )}
      </div>
    </div>
  );
}

