"use client";

import { useEffect } from "react";
import type { Company, LogoSettings } from "@/lib/types";

interface MetaTagsProps {
  profile: Company | null;
  canonicalUrl: string;
  logoSettings?: LogoSettings | null;
}

export default function MetaTags({ profile, canonicalUrl, logoSettings }: MetaTagsProps) {
  useEffect(() => {
    if (!profile) return;

    const companyName = profile.name || "Company";
    const description = profile.description
      ? profile.description.replace(/<[^>]+>/g, "").substring(0, 160)
      : `Explore career opportunities at ${companyName}. Find your perfect job match.`;

    // Update title
    document.title = `${companyName} - Career`;

    // Update or create meta tags
    const updateMetaTag = (name: string, content: string, property = false) => {
      const selector = property ? `property="${name}"` : `name="${name}"`;
      let meta = document.querySelector(`meta[${selector}]`);
      
      if (!meta) {
        meta = document.createElement("meta");
        if (property) {
          meta.setAttribute("property", name);
        } else {
          meta.setAttribute("name", name);
        }
        document.head.appendChild(meta);
      }
      meta.setAttribute("content", content);
    };

    // Standard meta tags
    updateMetaTag("description", description);
    updateMetaTag("keywords", [
      companyName,
      "company profile",
      "jobs",
      "careers",
      "recruitment",
      "Flowxtra",
      profile.company_type?.name,
      profile.number_employees ? `${profile.number_employees} employees` : "",
    ]
      .filter(Boolean)
      .join(", "));

    // Open Graph meta tags
    updateMetaTag("og:title", `${companyName} - Career`, true);
    updateMetaTag("og:description", description, true);
    updateMetaTag("og:type", "website", true);
    updateMetaTag("og:url", canonicalUrl, true);
    if (profile.logo) {
      updateMetaTag("og:image", profile.logo, true);
    }

    // Twitter Card meta tags
    updateMetaTag("twitter:card", "summary_large_image", true);
    updateMetaTag("twitter:title", `${companyName} - Career`, true);
    updateMetaTag("twitter:description", description, true);
    if (profile.logo) {
      updateMetaTag("twitter:image", profile.logo, true);
    }

    // Canonical URL
    let canonicalLink = document.querySelector('link[rel="canonical"]');
    if (!canonicalLink) {
      canonicalLink = document.createElement("link");
      canonicalLink.setAttribute("rel", "canonical");
      document.head.appendChild(canonicalLink);
    }
    canonicalLink.setAttribute("href", canonicalUrl);

    // Dynamic Favicon - Use Brand Kit favicon or company logo as fallback
    let faviconUrl: string | null = null;

    // Check Brand Kit favicon first
    if (logoSettings) {
      // Handle nested format
      if (logoSettings.favicon) {
        const favicon = logoSettings.favicon;
        const imageUrl = favicon.imageUrl || favicon.image_url;
        if (imageUrl) {
          faviconUrl = imageUrl.startsWith("http") ? imageUrl : imageUrl;
        }
      }
      // Handle flat snake_case format
      else if (logoSettings.favicon_url) {
        const imageUrl = logoSettings.favicon_url;
        faviconUrl = imageUrl.startsWith("http") ? imageUrl : imageUrl;
      }
    }

    // Fallback to profile logo
    if (!faviconUrl && profile.logo) {
      faviconUrl = profile.logo.startsWith("http") ? profile.logo : profile.logo;
    }

    if (faviconUrl) {
      // Remove existing favicon links
      const existingFavicons = document.querySelectorAll('link[rel="icon"], link[rel="shortcut icon"], link[rel="apple-touch-icon"]');
      // Convert NodeList to Array to avoid issues during iteration
      Array.from(existingFavicons).forEach((link) => {
        if (link && link.parentNode && link.parentNode === document.head && document.contains(link)) {
          try {
            link.remove();
          } catch (error) {
            // Ignore errors - element may have been removed already
          }
        }
      });

      // Add new favicon
      const faviconLink = document.createElement("link");
      faviconLink.setAttribute("rel", "icon");
      faviconLink.setAttribute("type", "image/png");
      faviconLink.setAttribute("href", faviconUrl);
      document.head.appendChild(faviconLink);

      // Add shortcut icon for older browsers
      const shortcutLink = document.createElement("link");
      shortcutLink.setAttribute("rel", "shortcut icon");
      shortcutLink.setAttribute("href", faviconUrl);
      document.head.appendChild(shortcutLink);

      // Add Apple touch icon for better mobile support
      const appleTouchIcon = document.createElement("link");
      appleTouchIcon.setAttribute("rel", "apple-touch-icon");
      appleTouchIcon.setAttribute("href", faviconUrl);
      document.head.appendChild(appleTouchIcon);
    }

    // JSON-LD Structured Data
    let jsonLdScript = document.getElementById("company-json-ld") as HTMLScriptElement | null;
    if (jsonLdScript && jsonLdScript.parentNode && document.contains(jsonLdScript)) {
      try {
        jsonLdScript.remove();
      } catch (error) {
        console.warn("Error removing JSON-LD script:", error);
      }
    }
    jsonLdScript = document.createElement("script");
    jsonLdScript.id = "company-json-ld";
    jsonLdScript.type = "application/ld+json";

    const jsonLd = {
      "@context": "https://schema.org",
      "@type": "Organization",
      name: companyName,
      description: description,
      url: canonicalUrl,
      ...(profile.logo && {
        logo: {
          "@type": "ImageObject",
          url: profile.logo,
        },
      }),
      ...(profile.website && {
        sameAs: [profile.website],
      }),
      ...(profile.number_employees && {
        numberOfEmployees: {
          "@type": "QuantitativeValue",
          value: profile.number_employees,
        },
      }),
      ...(profile.company_type && {
        industry: profile.company_type.name,
      }),
      ...(profile.benefits_company && profile.benefits_company.length > 0 && {
        employeeBenefits: profile.benefits_company.map((benefit) => benefit.name),
      }),
    };

    jsonLdScript.textContent = JSON.stringify(jsonLd);
    document.head.appendChild(jsonLdScript);
  }, [profile, canonicalUrl, logoSettings]);

  return null;
}

