"use client";

import { useEffect } from "react";

export interface JobMetaPayload {
  title: string;
  description: string;
  company?: {
    name?: string | null;
    logo?: string | null;
    website?: string | null;
  };
  company_office?: {
    address?: string | null;
    state?: { name?: string | null } | null;
    country?: { name?: string | null } | null;
    postal_code?: string | null;
  };
  job_type_job?: Array<{ job_type: { name?: string | null } }>;
  min_salary?: number | null;
  max_salary?: number | null;
  currency?: string | null;
  rate_salary?: string | null;
  hours_number?: number | null;
  benefits_job?: Array<{ name: string }>;
  skill_job?: Array<{ name: string }>;
  industry?: string | null;
  created_at?: string | null;
  updated_at?: string | null;
}

interface JobMetaTagsProps {
  job: JobMetaPayload;
  url: string;
}

export default function JobMetaTags({ job, url }: JobMetaTagsProps) {
  useEffect(() => {
    if (!job) return;

    const stripHtml = (html?: string | null) =>
      html ? html.replace(/<[^>]+>/g, "").trim() : "";

    const pageTitle =
      job.title || job.company?.name || "Job Opportunity";
    const cleanDescription =
      stripHtml(job.description) ||
      `Apply to ${job.title || "job"} at ${job.company?.name || "company"}.`;

    // Calculate validThrough (+35 days from updated_at or created_at)
    const baseDate = job.updated_at || job.created_at;
    const validThrough = baseDate
      ? new Date(
          new Date(baseDate).setDate(new Date(baseDate).getDate() + 35)
        ).toISOString()
      : undefined;

    const updateMeta = (name: string, content: string, property = false) => {
      const selector = property ? `property="${name}"` : `name="${name}"`;
      let meta = document.querySelector(`meta[${selector}]`);
      if (!meta) {
        meta = document.createElement("meta");
        if (property) meta.setAttribute("property", name);
        else meta.setAttribute("name", name);
        document.head.appendChild(meta);
      }
      meta.setAttribute("content", content);
    };

    document.title = pageTitle;

    updateMeta("description", cleanDescription);
    updateMeta(
      "keywords",
      [
        pageTitle,
        job.company?.name,
        job.industry,
        "job",
        "career",
      ]
        .filter(Boolean)
        .join(", ")
    );

    updateMeta("og:title", pageTitle, true);
    updateMeta("og:description", cleanDescription, true);
    updateMeta("og:type", "website", true);
    updateMeta("og:url", url, true);
    if (job.company?.logo) updateMeta("og:image", job.company.logo, true);

    updateMeta("twitter:card", "summary_large_image", true);
    updateMeta("twitter:title", pageTitle, true);
    updateMeta("twitter:description", cleanDescription, true);
    if (job.company?.logo) updateMeta("twitter:image", job.company.logo, true);

    let canonical = document.querySelector('link[rel="canonical"]');
    if (!canonical) {
      canonical = document.createElement("link");
      canonical.setAttribute("rel", "canonical");
      document.head.appendChild(canonical);
    }
    canonical.setAttribute("href", url);

    // Build JSON-LD JobPosting (موسع كما في Angular)
    const scriptId = "job-posting-json-ld";
    const existing = document.getElementById(scriptId) as HTMLScriptElement | null;
    if (existing) existing.remove();
    const jsonLdScript = document.createElement("script");
    jsonLdScript.id = scriptId;
    jsonLdScript.type = "application/ld+json";

    const countryAny = job.company_office?.country as any;
    const addressCountry =
      countryAny?.iso2?.toUpperCase?.() ||
      countryAny?.code?.toUpperCase?.() ||
      undefined;
    const addressObj =
      job.company_office?.address ||
      job.company_office?.state?.name ||
      job.company_office?.country?.name
        ? {
            "@type": "PostalAddress",
            streetAddress: job.company_office?.address || undefined,
            addressLocality: job.company_office?.state?.name || undefined,
            addressRegion: job.company_office?.state?.name || undefined,
            postalCode: job.company_office?.postal_code || undefined,
            addressCountry,
          }
        : {
            "@type": "PostalAddress",
            addressCountry,
          };

    const hiringOrgName = job.company?.name || "Organization";

    const jobPostingData: Record<string, any> = {
      "@context": "https://schema.org/",
      "@type": "JobPosting",
      title: pageTitle,
      description: stripHtml(job.description),
      datePosted: job.created_at ? new Date(job.created_at).toISOString() : undefined,
      validThrough,
      employmentType:
        job.job_type_job?.[0]?.job_type?.name || "FULL_TIME",
      hiringOrganization: {
        "@type": "Organization",
        name: hiringOrgName,
        logo: job.company?.logo || "",
        sameAs: job.company?.website,
        address: addressObj,
      },
      publisher: {
        "@type": "Organization",
        name: hiringOrgName,
        logo: job.company?.logo || "",
        sameAs: job.company?.website,
      },
      jobLocation: {
        "@type": "Place",
        address: addressObj,
      },
      baseSalary: {
        "@type": "MonetaryAmount",
        currency: job.currency || "USD",
        value: {
          "@type": "QuantitativeValue",
          minValue: job.min_salary,
          maxValue: job.max_salary,
          unitText: job.rate_salary,
        },
      },
      industry: job.industry,
      skills: job.skill_job?.map((s) => s.name).join(", "),
      workHours: job.hours_number ? `${job.hours_number} Hours per week` : undefined,
      jobBenefits: job.benefits_job?.map((b) => b.name).join(", "),
      url,
    };

    jsonLdScript.textContent = JSON.stringify(jobPostingData);
    document.head.appendChild(jsonLdScript);
  }, [job, url]);

  return null;
}



